<?php
/**
 * Transactions Management Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin', 'finance']);

$pageTitle = 'Manajemen Transaksi';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

$action = $_GET['action'] ?? 'list';
$transactionId = $_GET['id'] ?? 0;
$filterEventId = $_GET['event_id'] ?? '';
$filterType = $_GET['type'] ?? '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
        redirect('transactions.php');
    }
    
    $data = [
        'event_id' => !empty($_POST['event_id']) ? intval($_POST['event_id']) : null,
        'date' => $_POST['date'] ?? date('Y-m-d'),
        'type' => $_POST['type'] ?? 'expense',
        'category' => sanitizeInput($_POST['category'] ?? ''),
        'amount' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['amount'] ?? 0)),
        'description' => sanitizeInput($_POST['description'] ?? '')
    ];
    
    if (empty($data['category']) || $data['amount'] <= 0) {
        setFlash('error', 'Kategori dan jumlah wajib diisi');
    } else {
        // Generate transaction code
        $transCode = 'TRX' . date('Ymd') . strtoupper(substr(uniqid(), -6));
        
        if ($transactionId) {
            // Update existing transaction
            $stmt = $db->prepare("
                UPDATE transactions SET 
                    event_id = ?, date = ?, type = ?, category = ?, amount = ?, description = ?
                WHERE id = ? AND business_id = ?
            ");
            $stmt->execute([
                $data['event_id'], $data['date'], $data['type'], $data['category'],
                $data['amount'], $data['description'], $transactionId, $businessId
            ]);
            
            logActivity('update_transaction', "Updated transaction: {$transCode}");
            setFlash('success', 'Transaksi berhasil diperbarui');
        } else {
            // Create new transaction
            $stmt = $db->prepare("
                INSERT INTO transactions (
                    business_id, event_id, transaction_code, date, type, category, amount, description, created_by
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $businessId, $data['event_id'], $transCode, $data['date'],
                $data['type'], $data['category'], $data['amount'], $data['description'], $_SESSION['user_id']
            ]);
            
            logActivity('create_transaction', "Created transaction: {$transCode}");
            setFlash('success', 'Transaksi berhasil ditambahkan');
        }
        
        redirect('transactions.php');
    }
}

// Handle delete
if ($action === 'delete' && $transactionId) {
    if (!validateCSRFToken($_GET['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
    } else {
        $stmt = $db->prepare("DELETE FROM transactions WHERE id = ? AND business_id = ?");
        $stmt->execute([$transactionId, $businessId]);
        
        logActivity('delete_transaction', "Deleted transaction ID: {$transactionId}");
        setFlash('success', 'Transaksi berhasil dihapus');
    }
    redirect('transactions.php');
}

// Get transaction for edit
$transaction = null;
if ($transactionId && $action === 'edit') {
    $stmt = $db->prepare("SELECT * FROM transactions WHERE id = ? AND business_id = ?");
    $stmt->execute([$transactionId, $businessId]);
    $transaction = $stmt->fetch();
}

// Get events for dropdown
$stmt = $db->prepare("SELECT id, name FROM events WHERE business_id = ? ORDER BY name");
$stmt->execute([$businessId]);
$events = $stmt->fetchAll();

// Build query for transactions list
$query = "
    SELECT t.*, e.name as event_name
    FROM transactions t
    LEFT JOIN events e ON t.event_id = e.id
    WHERE t.business_id = ?
";
$params = [$businessId];

if ($filterEventId) {
    $query .= " AND t.event_id = ?";
    $params[] = $filterEventId;
}

if ($filterType) {
    $query .= " AND t.type = ?";
    $params[] = $filterType;
}

$query .= " ORDER BY t.date DESC, t.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$transactions = $stmt->fetchAll();

// Get summary
$summary = [
    'total_income' => array_sum(array_map(fn($t) => $t['type'] === 'income' ? $t['amount'] : 0, $transactions)),
    'total_expense' => array_sum(array_map(fn($t) => $t['type'] === 'expense' ? $t['amount'] : 0, $transactions)),
];
$summary['balance'] = $summary['total_income'] - $summary['total_expense'];

// Categories
$incomeCategories = ['Pendaftaran', 'Sponsorship', 'Penjualan Paket', 'Donasi', 'Lainnya'];
$expenseCategories = ['Hadiah', 'Piala', 'Desain', 'Promosi', 'Venue', 'Konsumsi', 'Operasional', 'Lainnya'];

// Monthly summary for chart
$stmt = $db->prepare("
    SELECT 
        DATE_FORMAT(date, '%Y-%m') as month,
        SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as income,
        SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as expense
    FROM transactions
    WHERE business_id = ? AND date >= DATE_SUB(CURRENT_DATE(), INTERVAL 6 MONTH)
    GROUP BY DATE_FORMAT(date, '%Y-%m')
    ORDER BY month
");
$stmt->execute([$businessId]);
$monthlyData = $stmt->fetchAll();
?>

<?php if ($action === 'create' || ($action === 'edit' && $transaction)): ?>
<!-- Create/Edit Form -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="bi bi-<?php echo $action === 'create' ? 'plus-circle' : 'pencil'; ?> me-2"></i>
            <?php echo $action === 'create' ? 'Tambah Transaksi' : 'Edit Transaksi'; ?>
        </h5>
    </div>
    <div class="card-body">
        <form method="POST" action="" class="needs-validation" novalidate>
            <?php echo csrfField(); ?>
            
            <div class="row g-3">
                <div class="col-md-6">
                    <label class="form-label">Tipe Transaksi <span class="text-danger">*</span></label>
                    <select class="form-select" name="type" id="transaction_type" required onchange="updateCategories()">
                        <option value="income" <?php echo ($transaction['type'] ?? '') === 'income' ? 'selected' : ''; ?>>Pemasukan (Income)</option>
                        <option value="expense" <?php echo ($transaction['type'] ?? 'expense') === 'expense' ? 'selected' : ''; ?>>Pengeluaran (Expense)</option>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Tanggal <span class="text-danger">*</span></label>
                    <input type="date" class="form-control" name="date" 
                           value="<?php echo $transaction['date'] ?? date('Y-m-d'); ?>" required>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Event (Opsional)</label>
                    <select class="form-select" name="event_id">
                        <option value="">Umum / Tidak terkait event</option>
                        <?php foreach ($events as $evt): ?>
                        <option value="<?php echo $evt['id']; ?>" <?php echo ($transaction['event_id'] ?? '') == $evt['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($evt['name']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Kategori <span class="text-danger">*</span></label>
                    <select class="form-select" name="category" id="category" required>
                        <option value="">Pilih Kategori</option>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Jumlah (Rp) <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" name="amount" 
                           value="<?php echo isset($transaction['amount']) ? formatRupiah($transaction['amount']) : ''; ?>"
                           onkeyup="formatCurrency(this)" required>
                </div>
                
                <div class="col-12">
                    <label class="form-label">Deskripsi</label>
                    <textarea class="form-control" name="description" rows="3"><?php echo htmlspecialchars($transaction['description'] ?? ''); ?></textarea>
                </div>
            </div>
            
            <div class="mt-4 d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-check-lg me-2"></i>Simpan
                </button>
                <a href="transactions.php" class="btn btn-secondary">
                    <i class="bi bi-x-lg me-2"></i>Batal
                </a>
            </div>
        </form>
    </div>
</div>

<script>
const incomeCategories = <?php echo json_encode($incomeCategories); ?>;
const expenseCategories = <?php echo json_encode($expenseCategories); ?>;
const currentCategory = '<?php echo $transaction['category'] ?? ''; ?>';

function updateCategories() {
    const type = document.getElementById('transaction_type').value;
    const categorySelect = document.getElementById('category');
    const categories = type === 'income' ? incomeCategories : expenseCategories;
    
    categorySelect.innerHTML = '<option value="">Pilih Kategori</option>';
    categories.forEach(cat => {
        const option = document.createElement('option');
        option.value = cat;
        option.textContent = cat;
        if (cat === currentCategory) option.selected = true;
        categorySelect.appendChild(option);
    });
}

// Initialize categories
updateCategories();
</script>

<?php else: ?>
<!-- Transactions List -->
<div class="row g-4 mb-4">
    <div class="col-md-4">
        <div class="stat-card success">
            <i class="bi bi-arrow-down-circle stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($summary['total_income']); ?></div>
            <div class="stat-label">Total Pemasukan</div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card danger">
            <i class="bi bi-arrow-up-circle stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($summary['total_expense']); ?></div>
            <div class="stat-label">Total Pengeluaran</div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card <?php echo $summary['balance'] >= 0 ? 'info' : 'warning'; ?>">
            <i class="bi bi-wallet2 stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($summary['balance']); ?></div>
            <div class="stat-label">Saldo</div>
        </div>
    </div>
</div>

<div class="row g-4 mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Grafik Pemasukan vs Pengeluaran</h5>
            </div>
            <div class="card-body">
                <canvas id="transactionChart" height="100"></canvas>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
        <h5 class="card-title mb-0">Daftar Transaksi</h5>
        <div class="d-flex gap-2 flex-wrap">
            <select class="form-select" style="width: 150px;" onchange="filterByType(this.value)">
                <option value="">Semua Tipe</option>
                <option value="income" <?php echo $filterType === 'income' ? 'selected' : ''; ?>>Pemasukan</option>
                <option value="expense" <?php echo $filterType === 'expense' ? 'selected' : ''; ?>>Pengeluaran</option>
            </select>
            <input type="text" id="searchTransaction" class="form-control" placeholder="Cari transaksi..." style="width: 200px;">
            <a href="transactions.php?action=create" class="btn btn-primary">
                <i class="bi bi-plus-lg me-2"></i>Tambah
            </a>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="transactionsTable">
                <thead>
                    <tr>
                        <th>Kode</th>
                        <th>Tanggal</th>
                        <th>Event</th>
                        <th>Kategori</th>
                        <th>Tipe</th>
                        <th>Jumlah</th>
                        <th class="no-print">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($transactions as $t): ?>
                    <tr>
                        <td><code><?php echo $t['transaction_code']; ?></code></td>
                        <td><?php echo formatDate($t['date']); ?></td>
                        <td><?php echo htmlspecialchars($t['event_name'] ?: '-'); ?></td>
                        <td><?php echo $t['category']; ?></td>
                        <td>
                            <span class="badge bg-<?php echo $t['type'] === 'income' ? 'success' : 'danger'; ?>">
                                <?php echo $t['type'] === 'income' ? 'Pemasukan' : 'Pengeluaran'; ?>
                            </span>
                        </td>
                        <td class="text-<?php echo $t['type'] === 'income' ? 'success' : 'danger'; ?>">
                            <?php echo $t['type'] === 'income' ? '+' : '-'; ?> <?php echo formatRupiah($t['amount']); ?>
                        </td>
                        <td class="no-print">
                            <div class="btn-group btn-group-sm">
                                <a href="transactions.php?action=edit&id=<?php echo $t['id']; ?>" class="btn btn-outline-primary" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <a href="transactions.php?action=delete&id=<?php echo $t['id']; ?>&csrf_token=<?php echo generateCSRFToken(); ?>" 
                                   class="btn btn-outline-danger" title="Hapus"
                                   onclick="return confirmDelete()">
                                    <i class="bi bi-trash"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($transactions)): ?>
                    <tr>
                        <td colspan="7" class="text-center py-5">
                            <i class="bi bi-cash-stack fs-1 text-muted"></i>
                            <p class="mt-3 text-muted">Belum ada transaksi</p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function filterByType(type) {
    const url = new URL(window.location.href);
    if (type) {
        url.searchParams.set('type', type);
    } else {
        url.searchParams.delete('type');
    }
    window.location.href = url.toString();
}

searchTable('searchTransaction', 'transactionsTable');

// Transaction Chart
const ctx = document.getElementById('transactionChart').getContext('2d');
new Chart(ctx, {
    type: 'bar',
    data: {
        labels: <?php echo json_encode(array_map(fn($d) => date('M Y', strtotime($d['month'] . '-01')), $monthlyData)); ?>,
        datasets: [{
            label: 'Pemasukan',
            data: <?php echo json_encode(array_map(fn($d) => $d['income'], $monthlyData)); ?>,
            backgroundColor: 'rgba(16, 185, 129, 0.8)'
        }, {
            label: 'Pengeluaran',
            data: <?php echo json_encode(array_map(fn($d) => $d['expense'], $monthlyData)); ?>,
            backgroundColor: 'rgba(239, 68, 68, 0.8)'
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: { position: 'top' }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return 'Rp ' + value.toLocaleString('id-ID');
                    }
                }
            }
        }
    }
});
</script>

<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
