<?php
/**
 * Settings Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin']);

$pageTitle = 'Pengaturan';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

$activeTab = $_GET['tab'] ?? 'general';

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
    } else {
        switch ($activeTab) {
            case 'general':
                // Update business info
                $stmt = $db->prepare("
                    UPDATE businesses SET 
                        name = ?, description = ?, address = ?, phone = ?, email = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    sanitizeInput($_POST['business_name'] ?? ''),
                    sanitizeInput($_POST['business_description'] ?? ''),
                    sanitizeInput($_POST['business_address'] ?? ''),
                    sanitizeInput($_POST['business_phone'] ?? ''),
                    sanitizeInput($_POST['business_email'] ?? ''),
                    $businessId
                ]);
                
                // Update settings
                $settings = ['app_name', 'currency', 'timezone', 'date_format', 'items_per_page'];
                foreach ($settings as $key) {
                    $stmt = $db->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = VALUES(value)");
                    $stmt->execute([$key, sanitizeInput($_POST[$key] ?? '')]);
                }
                
                setFlash('success', 'Pengaturan umum berhasil diperbarui');
                break;
                
            case 'users':
                // Add new user
                if (isset($_POST['add_user'])) {
                    $name = sanitizeInput($_POST['user_name'] ?? '');
                    $email = sanitizeInput($_POST['user_email'] ?? '');
                    $role = $_POST['user_role'] ?? 'event_manager';
                    $password = $_POST['user_password'] ?? '';
                    
                    if ($name && $email && $password) {
                        // Check if email exists
                        $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
                        $stmt->execute([$email]);
                        if ($stmt->fetch()) {
                            setFlash('error', 'Email sudah terdaftar');
                        } else {
                            $stmt = $db->prepare("
                                INSERT INTO users (name, email, password, role, is_active)
                                VALUES (?, ?, ?, ?, 1)
                            ");
                            $stmt->execute([$name, $email, encryptPassword($password), $role]);
                            
                            $newUserId = $db->lastInsertId();
                            
                            // Link to business
                            $stmt = $db->prepare("INSERT INTO business_users (business_id, user_id) VALUES (?, ?)");
                            $stmt->execute([$businessId, $newUserId]);
                            
                            setFlash('success', 'Pengguna berhasil ditambahkan');
                        }
                    }
                }
                break;
                
            case 'backup':
                // Create backup
                if (isset($_POST['create_backup'])) {
                    $backupDir = __DIR__ . '/backup/';
                    if (!is_dir($backupDir)) {
                        mkdir($backupDir, 0755, true);
                    }
                    
                    $filename = 'backup_' . date('Ymd_His') . '.sql';
                    $filepath = $backupDir . $filename;
                    
                    // Simple SQL dump (in production, use mysqldump command)
                    $tables = ['users', 'businesses', 'business_users', 'events', 'participants', 'transactions', 'event_metrics', 'settings'];
                    $sql = "-- Backup created on " . date('Y-m-d H:i:s') . "\n\n";
                    $sql .= "SET FOREIGN_KEY_CHECKS = 0;\n\n";
                    
                    foreach ($tables as $table) {
                        $sql .= "-- Table: {$table}\n";
                        $sql .= "DROP TABLE IF EXISTS {$table};\n";
                        
                        // Get create table
                        $stmt = $db->query("SHOW CREATE TABLE {$table}");
                        $create = $stmt->fetch();
                        $sql .= $create['Create Table'] . ";\n\n";
                        
                        // Get data
                        $stmt = $db->query("SELECT * FROM {$table}");
                        $rows = $stmt->fetchAll();
                        
                        if (!empty($rows)) {
                            $columns = array_keys($rows[0]);
                            $sql .= "INSERT INTO {$table} (" . implode(', ', $columns) . ") VALUES\n";
                            
                            $values = [];
                            foreach ($rows as $row) {
                                $rowValues = [];
                                foreach ($row as $value) {
                                    $rowValues[] = $value === null ? 'NULL' : $db->quote($value);
                                }
                                $values[] = '(' . implode(', ', $rowValues) . ')';
                            }
                            $sql .= implode(",\n", $values) . ";\n\n";
                        }
                    }
                    
                    $sql .= "SET FOREIGN_KEY_CHECKS = 1;\n";
                    
                    file_put_contents($filepath, $sql);
                    
                    // Save backup record
                    $stmt = $db->prepare("INSERT INTO backups (filename, file_path, file_size, created_by) VALUES (?, ?, ?, ?)");
                    $stmt->execute([$filename, $filepath, filesize($filepath), $_SESSION['user_id']]);
                    
                    setFlash('success', 'Backup berhasil dibuat: ' . $filename);
                }
                break;
        }
        
        redirect("settings.php?tab={$activeTab}");
    }
}

// Get current settings
$stmt = $db->query("SELECT * FROM settings");
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['key_name']] = $row['value'];
}

// Get business info
$stmt = $db->prepare("SELECT * FROM businesses WHERE id = ?");
$stmt->execute([$businessId]);
$business = $stmt->fetch();

// Get users
$stmt = $db->prepare("
    SELECT u.* FROM users u
    JOIN business_users bu ON u.id = bu.user_id
    WHERE bu.business_id = ?
    ORDER BY u.name
");
$stmt->execute([$businessId]);
$users = $stmt->fetchAll();

// Get backups
$stmt = $db->query("SELECT b.*, u.name as created_by_name FROM backups b LEFT JOIN users u ON b.created_by = u.id ORDER BY b.created_at DESC LIMIT 10");
$backups = $stmt->fetchAll();

$roles = [
    'owner' => 'Owner',
    'admin' => 'Admin',
    'finance' => 'Finance',
    'event_manager' => 'Event Manager'
];
?>

<div class="row">
    <div class="col-md-3">
        <div class="card">
            <div class="list-group list-group-flush">
                <a href="?tab=general" class="list-group-item list-group-item-action <?php echo $activeTab === 'general' ? 'active' : ''; ?>">
                    <i class="bi bi-gear me-2"></i>Umum
                </a>
                <a href="?tab=users" class="list-group-item list-group-item-action <?php echo $activeTab === 'users' ? 'active' : ''; ?>">
                    <i class="bi bi-people me-2"></i>Pengguna
                </a>
                <a href="?tab=backup" class="list-group-item list-group-item-action <?php echo $activeTab === 'backup' ? 'active' : ''; ?>">
                    <i class="bi bi-cloud-arrow-up me-2"></i>Backup & Restore
                </a>
            </div>
        </div>
    </div>
    
    <div class="col-md-9">
        <?php if ($activeTab === 'general'): ?>
        <!-- General Settings -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Pengaturan Umum</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="">
                    <?php echo csrfField(); ?>
                    
                    <h6 class="mb-3">Informasi Bisnis</h6>
                    <div class="row g-3 mb-4">
                        <div class="col-md-6">
                            <label class="form-label">Nama Bisnis</label>
                            <input type="text" class="form-control" name="business_name" 
                                   value="<?php echo htmlspecialchars($business['name'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email Bisnis</label>
                            <input type="email" class="form-control" name="business_email" 
                                   value="<?php echo htmlspecialchars($business['email'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Telepon</label>
                            <input type="text" class="form-control" name="business_phone" 
                                   value="<?php echo htmlspecialchars($business['phone'] ?? ''); ?>">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Alamat</label>
                            <textarea class="form-control" name="business_address" rows="2"><?php echo htmlspecialchars($business['address'] ?? ''); ?></textarea>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Deskripsi</label>
                            <textarea class="form-control" name="business_description" rows="2"><?php echo htmlspecialchars($business['description'] ?? ''); ?></textarea>
                        </div>
                    </div>
                    
                    <h6 class="mb-3">Pengaturan Aplikasi</h6>
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Nama Aplikasi</label>
                            <input type="text" class="form-control" name="app_name" 
                                   value="<?php echo htmlspecialchars($settings['app_name'] ?? APP_NAME); ?>">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Mata Uang</label>
                            <select class="form-select" name="currency">
                                <option value="IDR" <?php echo ($settings['currency'] ?? 'IDR') === 'IDR' ? 'selected' : ''; ?>>IDR (Rupiah)</option>
                                <option value="USD" <?php echo ($settings['currency'] ?? '') === 'USD' ? 'selected' : ''; ?>>USD (Dollar)</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Zona Waktu</label>
                            <select class="form-select" name="timezone">
                                <option value="Asia/Jakarta" <?php echo ($settings['timezone'] ?? 'Asia/Jakarta') === 'Asia/Jakarta' ? 'selected' : ''; ?>>Asia/Jakarta (WIB)</option>
                                <option value="Asia/Makassar" <?php echo ($settings['timezone'] ?? '') === 'Asia/Makassar' ? 'selected' : ''; ?>>Asia/Makassar (WITA)</option>
                                <option value="Asia/Jayapura" <?php echo ($settings['timezone'] ?? '') === 'Asia/Jayapura' ? 'selected' : ''; ?>>Asia/Jayapura (WIT)</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Format Tanggal</label>
                            <select class="form-select" name="date_format">
                                <option value="d M Y" <?php echo ($settings['date_format'] ?? 'd M Y') === 'd M Y' ? 'selected' : ''; ?>>01 Jan 2024</option>
                                <option value="d/m/Y" <?php echo ($settings['date_format'] ?? '') === 'd/m/Y' ? 'selected' : ''; ?>>01/01/2024</option>
                                <option value="Y-m-d" <?php echo ($settings['date_format'] ?? '') === 'Y-m-d' ? 'selected' : ''; ?>>2024-01-01</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Item per Halaman</label>
                            <input type="number" class="form-control" name="items_per_page" 
                                   value="<?php echo htmlspecialchars($settings['items_per_page'] ?? '20'); ?>">
                        </div>
                    </div>
                    
                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-lg me-2"></i>Simpan Pengaturan
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <?php elseif ($activeTab === 'users'): ?>
        <!-- User Management -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">Manajemen Pengguna</h5>
                <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addUserModal">
                    <i class="bi bi-plus-lg me-1"></i>Tambah Pengguna
                </button>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Nama</th>
                                <th>Email</th>
                                <th>Role</th>
                                <th>Status</th>
                                <th>Terdaftar</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $u): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($u['name']); ?></td>
                                <td><?php echo htmlspecialchars($u['email']); ?></td>
                                <td><span class="badge bg-secondary"><?php echo $roles[$u['role']] ?? $u['role']; ?></span></td>
                                <td>
                                    <span class="badge bg-<?php echo $u['is_active'] ? 'success' : 'danger'; ?>">
                                        <?php echo $u['is_active'] ? 'Aktif' : 'Nonaktif'; ?>
                                    </span>
                                </td>
                                <td><?php echo formatDate($u['created_at']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Add User Modal -->
        <div class="modal fade" id="addUserModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <form method="POST" action="">
                        <?php echo csrfField(); ?>
                        <div class="modal-header">
                            <h5 class="modal-title">Tambah Pengguna Baru</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Nama Lengkap</label>
                                <input type="text" class="form-control" name="user_name" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Email</label>
                                <input type="email" class="form-control" name="user_email" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Password</label>
                                <input type="password" class="form-control" name="user_password" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Role</label>
                                <select class="form-select" name="user_role">
                                    <?php foreach ($roles as $key => $label): ?>
                                    <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                            <button type="submit" name="add_user" class="btn btn-primary">Simpan</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <?php elseif ($activeTab === 'backup'): ?>
        <!-- Backup & Restore -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Backup & Restore</h5>
            </div>
            <div class="card-body">
                <div class="row g-4">
                    <div class="col-md-6">
                        <div class="card border">
                            <div class="card-body text-center">
                                <i class="bi bi-cloud-arrow-up fs-1 text-primary mb-3"></i>
                                <h6>Backup Database</h6>
                                <p class="text-muted small">Buat backup lengkap database saat ini</p>
                                <form method="POST" action="">
                                    <?php echo csrfField(); ?>
                                    <button type="submit" name="create_backup" class="btn btn-primary">
                                        <i class="bi bi-download me-2"></i>Buat Backup
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card border">
                            <div class="card-body text-center">
                                <i class="bi bi-cloud-arrow-down fs-1 text-success mb-3"></i>
                                <h6>Restore Database</h6>
                                <p class="text-muted small">Pulihkan data dari file backup</p>
                                <div class="alert alert-warning small">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    Fitur restore dilakukan melalui phpMyAdmin
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <h6 class="mt-4 mb-3">Riwayat Backup</h6>
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>File</th>
                                <th>Ukuran</th>
                                <th>Dibuat Oleh</th>
                                <th>Tanggal</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($backups as $backup): ?>
                            <tr>
                                <td><?php echo $backup['filename']; ?></td>
                                <td><?php echo number_format($backup['file_size'] / 1024, 2); ?> KB</td>
                                <td><?php echo htmlspecialchars($backup['created_by_name'] ?? 'System'); ?></td>
                                <td><?php echo formatDate($backup['created_at'], 'd M Y H:i'); ?></td>
                                <td>
                                    <a href="backup/<?php echo $backup['filename']; ?>" class="btn btn-sm btn-success" download>
                                        <i class="bi bi-download"></i>
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($backups)): ?>
                            <tr>
                                <td colspan="5" class="text-center text-muted">Belum ada backup</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>
