<?php
/**
 * Reports Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin', 'finance']);

$pageTitle = 'Laporan';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

// Get filter parameters
$reportType = $_GET['report_type'] ?? 'financial';
$dateFrom = $_GET['date_from'] ?? date('Y-m-01');
$dateTo = $_GET['date_to'] ?? date('Y-m-t');
$eventId = $_GET['event_id'] ?? '';

// Get events for filter
$stmt = $db->prepare("SELECT id, name FROM events WHERE business_id = ? ORDER BY name");
$stmt->execute([$businessId]);
$events = $stmt->fetchAll();

$reportData = [];

if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['generate'])) {
    switch ($reportType) {
        case 'financial':
            // Financial report
            $query = "
                SELECT 
                    DATE_FORMAT(t.date, '%Y-%m-%d') as date,
                    t.transaction_code,
                    t.type,
                    t.category,
                    t.amount,
                    t.description,
                    e.name as event_name
                FROM transactions t
                LEFT JOIN events e ON t.event_id = e.id
                WHERE t.business_id = ? AND t.date BETWEEN ? AND ?
            ";
            $params = [$businessId, $dateFrom, $dateTo];
            
            if ($eventId) {
                $query .= " AND t.event_id = ?";
                $params[] = $eventId;
            }
            
            $query .= " ORDER BY t.date DESC";
            
            $stmt = $db->prepare($query);
            $stmt->execute($params);
            $reportData = $stmt->fetchAll();
            break;
            
        case 'participants':
            // Participants report
            $query = "
                SELECT 
                    p.registration_code,
                    p.name,
                    p.school,
                    p.class,
                    p.phone,
                    p.payment_status,
                    p.paid_amount,
                    p.registered_at,
                    e.name as event_name
                FROM participants p
                JOIN events e ON p.event_id = e.id
                WHERE e.business_id = ? AND DATE(p.registered_at) BETWEEN ? AND ?
            ";
            $params = [$businessId, $dateFrom, $dateTo];
            
            if ($eventId) {
                $query .= " AND p.event_id = ?";
                $params[] = $eventId;
            }
            
            $query .= " ORDER BY p.registered_at DESC";
            
            $stmt = $db->prepare($query);
            $stmt->execute($params);
            $reportData = $stmt->fetchAll();
            break;
            
        case 'events':
            // Events report
            $stmt = $db->prepare("
                SELECT 
                    e.name,
                    e.type,
                    e.start_date,
                    e.end_date,
                    e.capacity,
                    e.price,
                    e.status,
                    COUNT(p.id) as participant_count,
                    SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as revenue
                FROM events e
                LEFT JOIN participants p ON e.id = p.event_id
                WHERE e.business_id = ? AND e.start_date BETWEEN ? AND ?
                GROUP BY e.id
                ORDER BY e.start_date DESC
            ");
            $stmt->execute([$businessId, $dateFrom, $dateTo]);
            $reportData = $stmt->fetchAll();
            break;
    }
    
    logActivity('generate_report', "Generated {$reportType} report");
}

// Handle export
if (isset($_GET['export'])) {
    $exportType = $_GET['export'];
    $filename = "laporan_{$reportType}_" . date('Ymd') . ".{$exportType}";
    
    if ($exportType === 'csv') {
        header('Content-Type: text/csv; charset=utf-8');
        header("Content-Disposition: attachment; filename={$filename}");
        
        $output = fopen('php://output', 'w');
        
        // Add BOM for Excel
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // Headers
        if (!empty($reportData)) {
            fputcsv($output, array_keys($reportData[0]));
            
            // Data
            foreach ($reportData as $row) {
                fputcsv($output, $row);
            }
        }
        
        fclose($output);
        exit;
    }
}
?>

<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">Filter Laporan</h5>
    </div>
    <div class="card-body">
        <form method="GET" action="" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Jenis Laporan</label>
                <select class="form-select" name="report_type">
                    <option value="financial" <?php echo $reportType === 'financial' ? 'selected' : ''; ?>>Keuangan</option>
                    <option value="participants" <?php echo $reportType === 'participants' ? 'selected' : ''; ?>>Peserta</option>
                    <option value="events" <?php echo $reportType === 'events' ? 'selected' : ''; ?>>Event</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label">Dari Tanggal</label>
                <input type="date" class="form-control" name="date_from" value="<?php echo $dateFrom; ?>">
            </div>
            
            <div class="col-md-3">
                <label class="form-label">Sampai Tanggal</label>
                <input type="date" class="form-control" name="date_to" value="<?php echo $dateTo; ?>">
            </div>
            
            <div class="col-md-3">
                <label class="form-label">Event (Opsional)</label>
                <select class="form-select" name="event_id">
                    <option value="">Semua Event</option>
                    <?php foreach ($events as $evt): ?>
                    <option value="<?php echo $evt['id']; ?>" <?php echo $eventId == $evt['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($evt['name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-12">
                <button type="submit" name="generate" class="btn btn-primary">
                    <i class="bi bi-search me-2"></i>Generate Laporan
                </button>
            </div>
        </form>
    </div>
</div>

<?php if (!empty($reportData)): ?>
<div class="card mt-4">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            Hasil Laporan 
            <span class="badge bg-secondary"><?php echo count($reportData); ?> data</span>
        </h5>
        <div class="btn-group">
            <a href="<?php echo $_SERVER['REQUEST_URI']; ?>&export=csv" class="btn btn-success btn-sm">
                <i class="bi bi-file-earmark-excel me-1"></i>Export CSV
            </a>
            <button type="button" class="btn btn-secondary btn-sm" onclick="printPage()">
                <i class="bi bi-printer me-1"></i>Print
            </button>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="reportTable">
                <thead>
                    <tr>
                        <?php foreach (array_keys($reportData[0]) as $header): ?>
                        <th><?php echo ucwords(str_replace('_', ' ', $header)); ?></th>
                        <?php endforeach; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($reportData as $row): ?>
                    <tr>
                        <?php foreach ($row as $key => $value): ?>
                        <td>
                            <?php 
                            if (strpos($key, 'amount') !== false || strpos($key, 'revenue') !== false || strpos($key, 'price') !== false) {
                                echo formatRupiah($value);
                            } elseif (strpos($key, 'date') !== false || strpos($key, 'at') !== false) {
                                echo $value ? formatDate($value) : '-';
                            } elseif ($key === 'status') {
                                echo '<span class="badge badge-' . $value . '">' . ucfirst($value) . '</span>';
                            } elseif ($key === 'payment_status') {
                                echo '<span class="badge badge-' . $value . '">' . ucfirst($value) . '</span>';
                            } elseif ($key === 'type') {
                                echo '<span class="badge bg-' . ($value === 'income' ? 'success' : 'danger') . '">' . ($value === 'income' ? 'Pemasukan' : 'Pengeluaran') . '</span>';
                            } else {
                                echo htmlspecialchars($value ?: '-');
                            }
                            ?>
                        </td>
                        <?php endforeach; ?>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php elseif (isset($_GET['generate'])): ?>
<div class="alert alert-info mt-4">
    <i class="bi bi-info-circle me-2"></i>Tidak ada data untuk periode yang dipilih.
</div>
<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
