<?php
/**
 * Event Recommendations Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin']);

$pageTitle = 'Rekomendasi Event';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

// Get draft events for analysis
$stmt = $db->prepare("
    SELECT e.*, 
           AVG(em.margin_percent) as avg_margin_similar,
           AVG(em.total_participants) as avg_participants_similar
    FROM events e
    LEFT JOIN events e2 ON e.type = e2.type AND e2.status = 'selesai' AND e2.id != e.id
    LEFT JOIN event_metrics em ON e2.id = em.event_id
    WHERE e.business_id = ? AND e.status = 'draft'
    GROUP BY e.id
");
$stmt->execute([$businessId]);
$draftEvents = $stmt->fetchAll();

// Calculate recommendations
$recommendations = [];

foreach ($draftEvents as $event) {
    // Get historical data for similar events
    $stmt = $db->prepare("
        SELECT 
            AVG(em.total_participants) as avg_participants,
            AVG(em.margin_percent) as avg_margin,
            AVG(em.roi) as avg_roi,
            COUNT(*) as count
        FROM events e
        JOIN event_metrics em ON e.id = em.event_id
        WHERE e.business_id = ? AND e.type = ? AND e.status = 'selesai'
    ");
    $stmt->execute([$businessId, $event['type']]);
    $historical = $stmt->fetch();
    
    // Get overlapping events (conflict score)
    $stmt = $db->prepare("
        SELECT COUNT(*) as count
        FROM events
        WHERE business_id = ? AND id != ? AND status IN ('draft', 'dibuka', 'tertutup')
        AND (
            (start_date <= ? AND end_date >= ?) OR
            (start_date <= ? AND end_date >= ?)
        )
    ");
    $stmt->execute([$businessId, $event['id'], $event['end_date'], $event['start_date'], $event['start_date'], $event['end_date']]);
    $conflictCount = $stmt->fetch()['count'];
    
    // Calculate metrics
    $historicalParticipants = $historical['avg_participants'] ?? 50;
    $historicalMargin = $historical['avg_margin'] ?? 20;
    $historicalAttendance = 0.8; // 80% default attendance rate
    
    // Growth factor based on marketing reach
    $growthFactor = min(1 + ($event['marketing_reach'] / 10000), 1.5);
    
    $expectedParticipants = round($historicalParticipants * $growthFactor);
    $expectedRevenue = $expectedParticipants * $event['price'];
    $expectedCost = $event['fixed_cost'] + ($event['variable_cost_per_participant'] * $expectedParticipants) + $event['marketing_cost'] + $event['operational_cost'];
    $expectedMargin = $expectedRevenue - $expectedCost;
    $expectedMarginPercent = $expectedRevenue > 0 ? ($expectedMargin / $expectedRevenue) * 100 : 0;
    
    // Normalize values (0-1 scale)
    $normParticipants = min($expectedParticipants / $event['capacity'], 1);
    $normMargin = max(min($expectedMarginPercent / 50, 1), -1);
    $normAttendance = $historicalAttendance;
    $normMarketing = min($event['marketing_reach'] / 10000, 1);
    $normConflict = min($conflictCount / 3, 1);
    
    // Weights
    $w1 = 0.30; // expected_participants
    $w2 = 0.30; // expected_margin_percent
    $w3 = 0.20; // historical_attendance_rate
    $w4 = 0.15; // marketing_reach
    $w5 = 0.05; // conflict_score (penalty)
    
    // Calculate score
    $score = ($w1 * $normParticipants) + 
             ($w2 * $normMargin) + 
             ($w3 * $normAttendance) + 
             ($w4 * $normMarketing) - 
             ($w5 * $normConflict);
    
    // Generate reasons
    $reasons = [];
    if ($expectedMarginPercent > 30) $reasons[] = ['type' => 'success', 'text' => 'Margin tinggi (' . round($expectedMarginPercent, 1) . '%)'];
    if ($expectedMarginPercent < 10 && $expectedMarginPercent > 0) $reasons[] = ['type' => 'warning', 'text' => 'Margin rendah'];
    if ($expectedMarginPercent < 0) $reasons[] = ['type' => 'danger', 'text' => 'Prediksi rugi!'];
    if ($event['fixed_cost'] < 2000000) $reasons[] = ['type' => 'success', 'text' => 'Biaya tetap rendah'];
    if ($normParticipants > 0.8) $reasons[] = ['type' => 'success', 'text' => 'Kapasitas sesuai'];
    if ($conflictCount > 0) $reasons[] = ['type' => 'warning', 'text' => 'Bentrok dengan ' . $conflictCount . ' event'];
    if ($event['marketing_reach'] > 5000) $reasons[] = ['type' => 'success', 'text' => 'Jangkauan marketing baik'];
    
    // Break-even calculation
    $breakEvenParticipants = $event['price'] > 0 ? ceil($expectedCost / $event['price']) : 0;
    
    $recommendations[] = [
        'event' => $event,
        'score' => max(0, round($score * 100, 1)),
        'expected_participants' => $expectedParticipants,
        'expected_revenue' => $expectedRevenue,
        'expected_cost' => $expectedCost,
        'expected_margin' => $expectedMargin,
        'expected_margin_percent' => round($expectedMarginPercent, 1),
        'break_even_participants' => $breakEvenParticipants,
        'conflict_count' => $conflictCount,
        'historical_count' => $historical['count'] ?? 0,
        'reasons' => $reasons
    ];
}

// Sort by score descending
usort($recommendations, fn($a, $b) => $b['score'] <=> $a['score']);

// Get completed events for comparison
$stmt = $db->prepare("
    SELECT e.*, em.margin_percent, em.total_participants, em.total_revenue
    FROM events e
    JOIN event_metrics em ON e.id = em.event_id
    WHERE e.business_id = ? AND e.status = 'selesai'
    ORDER BY em.gross_margin DESC
    LIMIT 5
");
$stmt->execute([$businessId]);
$completedEvents = $stmt->fetchAll();

logActivity('view_recommendations', 'User viewed event recommendations');
?>

<div class="row g-4 mb-4">
    <div class="col-12">
        <div class="alert alert-info">
            <i class="bi bi-info-circle-fill me-2"></i>
            <strong>Sistem Rekomendasi Event</strong> - Skor dihitung berdasarkan: ekspektasi peserta (30%), margin (30%), 
            tingkat kehadiran historis (20%), jangkauan marketing (15%), dan konflik jadwal (5%).
        </div>
    </div>
</div>

<?php if (empty($recommendations)): ?>
<div class="card">
    <div class="card-body text-center py-5">
        <i class="bi bi-lightbulb fs-1 text-muted"></i>
        <h5 class="mt-3">Tidak Ada Event Draft</h5>
        <p class="text-muted">Buat event baru dengan status "Draft" untuk melihat rekomendasi.</p>
        <a href="events.php?action=create" class="btn btn-primary">Buat Event Baru</a>
    </div>
</div>
<?php else: ?>

<div class="row g-4">
    <?php foreach ($recommendations as $rec): ?>
    <div class="col-lg-6">
        <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0"><?php echo htmlspecialchars($rec['event']['name']); ?></h5>
                <span class="badge bg-<?php echo $rec['score'] >= 70 ? 'success' : ($rec['score'] >= 50 ? 'warning' : 'danger'); ?> fs-6">
                    Skor: <?php echo $rec['score']; ?>
                </span>
            </div>
            <div class="card-body">
                <div class="row g-3 mb-3">
                    <div class="col-6">
                        <small class="text-muted">Ekspektasi Peserta</small>
                        <p class="mb-0 fw-medium"><?php echo number_format($rec['expected_participants']); ?> / <?php echo $rec['event']['capacity']; ?></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Ekspektasi Revenue</small>
                        <p class="mb-0 fw-medium"><?php echo formatRupiah($rec['expected_revenue']); ?></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Estimasi Biaya</small>
                        <p class="mb-0 fw-medium"><?php echo formatRupiah($rec['expected_cost']); ?></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Ekspektasi Margin</small>
                        <p class="mb-0 fw-medium text-<?php echo $rec['expected_margin'] >= 0 ? 'success' : 'danger'; ?>">
                            <?php echo formatRupiah($rec['expected_margin']); ?> (<?php echo $rec['expected_margin_percent']; ?>%)
                        </p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Break-even Peserta</small>
                        <p class="mb-0 fw-medium"><?php echo number_format($rec['break_even_participants']); ?> orang</p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Data Historis</small>
                        <p class="mb-0 fw-medium"><?php echo $rec['historical_count']; ?> event serupa</p>
                    </div>
                </div>
                
                <hr>
                
                <h6 class="mb-2">Analisis:</h6>
                <div class="d-flex flex-wrap gap-2">
                    <?php foreach ($rec['reasons'] as $reason): ?>
                    <span class="badge bg-<?php echo $reason['type']; ?>"><?php echo $reason['text']; ?></span>
                    <?php endforeach; ?>
                </div>
                
                <?php if ($rec['expected_margin'] < 0): ?>
                <div class="alert alert-danger mt-3 mb-0">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i>
                    <strong>Perhatian!</strong> Event ini diprediksi akan merugi. 
                    Pertimbangkan untuk menaikkan harga atau mengurangi biaya.
                </div>
                <?php endif; ?>
            </div>
            <div class="card-footer">
                <div class="d-flex gap-2">
                    <a href="events.php?action=edit&id=<?php echo $rec['event']['id']; ?>" class="btn btn-primary btn-sm">
                        <i class="bi bi-pencil me-1"></i>Edit Event
                    </a>
                    <a href="events.php?action=view&id=<?php echo $rec['event']['id']; ?>" class="btn btn-outline-secondary btn-sm">
                        <i class="bi bi-eye me-1"></i>Detail
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<?php endif; ?>

<?php if (!empty($completedEvents)): ?>
<div class="card mt-4">
    <div class="card-header">
        <h5 class="card-title mb-0">Event Selesai (Referensi)</h5>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead>
                    <tr>
                        <th>Event</th>
                        <th>Peserta</th>
                        <th>Revenue</th>
                        <th>Margin %</th>
                        <th>Performa</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($completedEvents as $evt): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($evt['name']); ?></td>
                        <td><?php echo number_format($evt['total_participants']); ?></td>
                        <td><?php echo formatRupiah($evt['total_revenue']); ?></td>
                        <td>
                            <span class="badge bg-<?php echo $evt['margin_percent'] >= 30 ? 'success' : ($evt['margin_percent'] >= 10 ? 'warning' : 'danger'); ?>">
                                <?php echo number_format($evt['margin_percent'], 1); ?>%
                            </span>
                        </td>
                        <td>
                            <div class="progress" style="height: 6px;">
                                <div class="progress-bar bg-success" style="width: <?php echo min($evt['margin_percent'], 100); ?>%"></div>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
