<?php
/**
 * Participants Management Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin', 'event_manager']);

$pageTitle = 'Manajemen Peserta';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

$action = $_GET['action'] ?? 'list';
$participantId = $_GET['id'] ?? 0;
$filterEventId = $_GET['event_id'] ?? '';
$filterStatus = $_GET['status'] ?? '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
        redirect('participants.php');
    }
    
    $data = [
        'event_id' => intval($_POST['event_id'] ?? 0),
        'name' => sanitizeInput($_POST['name'] ?? ''),
        'email' => sanitizeInput($_POST['email'] ?? ''),
        'phone' => sanitizeInput($_POST['phone'] ?? ''),
        'school' => sanitizeInput($_POST['school'] ?? ''),
        'class' => sanitizeInput($_POST['class'] ?? ''),
        'address' => sanitizeInput($_POST['address'] ?? ''),
        'package' => sanitizeInput($_POST['package'] ?? 'Reguler'),
        'payment_status' => $_POST['payment_status'] ?? 'pending',
        'paid_amount' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['paid_amount'] ?? 0)),
        'payment_method' => $_POST['payment_method'] ?? '',
        'notes' => sanitizeInput($_POST['notes'] ?? '')
    ];
    
    if (empty($data['event_id']) || empty($data['name']) || empty($data['phone'])) {
        setFlash('error', 'Event, nama, dan nomor telepon wajib diisi');
    } else {
        // Generate registration code
        $regCode = 'REG' . date('Y') . strtoupper(substr(uniqid(), -6));
        
        if ($participantId) {
            // Update existing participant
            $oldParticipant = $db->prepare("SELECT payment_status FROM participants WHERE id = ?");
            $oldParticipant->execute([$participantId]);
            $oldStatus = $oldParticipant->fetch()['payment_status'] ?? 'pending';
            
            $stmt = $db->prepare("
                UPDATE participants SET 
                    event_id = ?, name = ?, email = ?, phone = ?, school = ?, class = ?,
                    address = ?, package = ?, payment_status = ?, paid_amount = ?,
                    payment_method = ?, notes = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $data['event_id'], $data['name'], $data['email'], $data['phone'],
                $data['school'], $data['class'], $data['address'], $data['package'],
                $data['payment_status'], $data['paid_amount'], $data['payment_method'],
                $data['notes'], $participantId
            ]);
            
            // Update payment date if status changed to paid
            if ($data['payment_status'] === 'paid' && $oldStatus !== 'paid') {
                $stmt = $db->prepare("UPDATE participants SET payment_date = NOW() WHERE id = ?");
                $stmt->execute([$participantId]);
            }
            
            // Update event metrics
            updateEventMetrics($data['event_id']);
            
            logActivity('update_participant', "Updated participant: {$data['name']}");
            setFlash('success', 'Peserta berhasil diperbarui');
        } else {
            // Create new participant
            $stmt = $db->prepare("
                INSERT INTO participants (
                    event_id, registration_code, name, email, phone, school, class,
                    address, package, payment_status, paid_amount, payment_method, notes
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $data['event_id'], $regCode, $data['name'], $data['email'], $data['phone'],
                $data['school'], $data['class'], $data['address'], $data['package'],
                $data['payment_status'], $data['paid_amount'], $data['payment_method'], $data['notes']
            ]);
            
            // Update event metrics
            updateEventMetrics($data['event_id']);
            
            logActivity('create_participant', "Created participant: {$data['name']} ({$regCode})");
            setFlash('success', 'Peserta berhasil ditambahkan. Kode: ' . $regCode);
        }
        
        redirect('participants.php' . ($filterEventId ? '?event_id=' . $filterEventId : ''));
    }
}

// Handle delete
if ($action === 'delete' && $participantId) {
    if (!validateCSRFToken($_GET['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
    } else {
        // Get event_id before delete
        $stmt = $db->prepare("SELECT event_id, name FROM participants WHERE id = ?");
        $stmt->execute([$participantId]);
        $participant = $stmt->fetch();
        
        if ($participant) {
            $stmt = $db->prepare("DELETE FROM participants WHERE id = ?");
            $stmt->execute([$participantId]);
            
            // Update event metrics
            updateEventMetrics($participant['event_id']);
            
            logActivity('delete_participant', "Deleted participant: {$participant['name']}");
            setFlash('success', 'Peserta berhasil dihapus');
        }
    }
    redirect('participants.php');
}

// Handle attendance
if ($action === 'attend' && $participantId) {
    $stmt = $db->prepare("UPDATE participants SET attended = 1, attendance_time = NOW() WHERE id = ?");
    $stmt->execute([$participantId]);
    setFlash('success', 'Kehadiran dicatat');
    redirect('participants.php');
}

// Helper function to update event metrics
function updateEventMetrics($eventId) {
    $db = getDB();
    
    $stmt = $db->prepare("
        SELECT 
            e.fixed_cost, e.variable_cost_per_participant,
            COUNT(p.id) as total_participants,
            SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as registration_revenue
        FROM events e
        LEFT JOIN participants p ON e.id = p.event_id
        WHERE e.id = ?
        GROUP BY e.id
    ");
    $stmt->execute([$eventId]);
    $data = $stmt->fetch();
    
    if ($data) {
        $totalRevenue = $data['registration_revenue'];
        $totalCost = $data['fixed_cost'] + ($data['variable_cost_per_participant'] * $data['total_participants']);
        $grossMargin = $totalRevenue - $totalCost;
        $marginPercent = $totalRevenue > 0 ? ($grossMargin / $totalRevenue) * 100 : 0;
        $roi = $totalCost > 0 ? ($grossMargin / $totalCost) * 100 : 0;
        $profitPerParticipant = $data['total_participants'] > 0 ? $grossMargin / $data['total_participants'] : 0;
        
        $stmt = $db->prepare("
            INSERT INTO event_metrics (event_id, total_participants, total_revenue, total_cost, gross_margin, margin_percent, roi, profit_per_participant)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                total_participants = VALUES(total_participants),
                total_revenue = VALUES(total_revenue),
                total_cost = VALUES(total_cost),
                gross_margin = VALUES(gross_margin),
                margin_percent = VALUES(margin_percent),
                roi = VALUES(roi),
                profit_per_participant = VALUES(profit_per_participant)
        ");
        $stmt->execute([
            $eventId, $data['total_participants'], $totalRevenue, $totalCost,
            $grossMargin, $marginPercent, $roi, $profitPerParticipant
        ]);
    }
}

// Get participant for edit
$participant = null;
if ($participantId && $action === 'edit') {
    $stmt = $db->prepare("SELECT * FROM participants WHERE id = ?");
    $stmt->execute([$participantId]);
    $participant = $stmt->fetch();
}

// Get events for dropdown
$stmt = $db->prepare("SELECT id, name, price FROM events WHERE business_id = ? ORDER BY name");
$stmt->execute([$businessId]);
$events = $stmt->fetchAll();

// Build query for participants list
$query = "
    SELECT p.*, e.name as event_name, e.price as event_price
    FROM participants p
    JOIN events e ON p.event_id = e.id
    WHERE e.business_id = ?
";
$params = [$businessId];

if ($filterEventId) {
    $query .= " AND p.event_id = ?";
    $params[] = $filterEventId;
}

if ($filterStatus) {
    $query .= " AND p.payment_status = ?";
    $params[] = $filterStatus;
}

$query .= " ORDER BY p.registered_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$participants = $stmt->fetchAll();

// Get summary
$summary = [
    'total' => count($participants),
    'paid' => count(array_filter($participants, fn($p) => $p['payment_status'] === 'paid')),
    'pending' => count(array_filter($participants, fn($p) => $p['payment_status'] === 'pending')),
    'total_revenue' => array_sum(array_map(fn($p) => $p['paid_amount'], array_filter($participants, fn($p) => $p['payment_status'] === 'paid')))
];

$paymentMethods = ['Transfer Bank', 'QRIS', 'Cash', 'E-Wallet'];
$packages = ['Reguler', 'Premium', 'VIP', 'Early Bird'];
?>

<?php if ($action === 'create' || ($action === 'edit' && $participant)): ?>
<!-- Create/Edit Form -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="bi bi-<?php echo $action === 'create' ? 'plus-circle' : 'pencil'; ?> me-2"></i>
            <?php echo $action === 'create' ? 'Tambah Peserta' : 'Edit Peserta'; ?>
        </h5>
    </div>
    <div class="card-body">
        <form method="POST" action="" class="needs-validation" novalidate>
            <?php echo csrfField(); ?>
            
            <div class="row g-3">
                <div class="col-md-6">
                    <label class="form-label">Event <span class="text-danger">*</span></label>
                    <select class="form-select" name="event_id" required>
                        <option value="">Pilih Event</option>
                        <?php foreach ($events as $evt): ?>
                        <option value="<?php echo $evt['id']; ?>" 
                                data-price="<?php echo $evt['price']; ?>"
                                <?php echo (($participant['event_id'] ?? $filterEventId) == $evt['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($evt['name']); ?> (<?php echo formatRupiah($evt['price']); ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Paket</label>
                    <select class="form-select" name="package">
                        <?php foreach ($packages as $pkg): ?>
                        <option value="<?php echo $pkg; ?>" <?php echo ($participant['package'] ?? 'Reguler') === $pkg ? 'selected' : ''; ?>>
                            <?php echo $pkg; ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Nama Lengkap <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" name="name" 
                           value="<?php echo htmlspecialchars($participant['name'] ?? ''); ?>" required>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Email</label>
                    <input type="email" class="form-control" name="email" 
                           value="<?php echo htmlspecialchars($participant['email'] ?? ''); ?>">
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Nomor Telepon <span class="text-danger">*</span></label>
                    <input type="tel" class="form-control" name="phone" 
                           value="<?php echo htmlspecialchars($participant['phone'] ?? ''); ?>" required>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Sekolah/Institusi</label>
                    <input type="text" class="form-control" name="school" 
                           value="<?php echo htmlspecialchars($participant['school'] ?? ''); ?>">
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Kelas/Departemen</label>
                    <input type="text" class="form-control" name="class" 
                           value="<?php echo htmlspecialchars($participant['class'] ?? ''); ?>">
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Status Pembayaran</label>
                    <select class="form-select" name="payment_status" id="payment_status">
                        <option value="pending" <?php echo ($participant['payment_status'] ?? 'pending') === 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="paid" <?php echo ($participant['payment_status'] ?? '') === 'paid' ? 'selected' : ''; ?>>Paid</option>
                        <option value="refunded" <?php echo ($participant['payment_status'] ?? '') === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Jumlah Dibayar (Rp)</label>
                    <input type="text" class="form-control" name="paid_amount" id="paid_amount"
                           value="<?php echo isset($participant['paid_amount']) ? formatRupiah($participant['paid_amount']) : 'Rp 0'; ?>"
                           onkeyup="formatCurrency(this)">
                </div>
                
                <div class="col-md-6">
                    <label class="form-label">Metode Pembayaran</label>
                    <select class="form-select" name="payment_method">
                        <option value="">Pilih Metode</option>
                        <?php foreach ($paymentMethods as $method): ?>
                        <option value="<?php echo $method; ?>" <?php echo ($participant['payment_method'] ?? '') === $method ? 'selected' : ''; ?>>
                            <?php echo $method; ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-12">
                    <label class="form-label">Alamat</label>
                    <textarea class="form-control" name="address" rows="2"><?php echo htmlspecialchars($participant['address'] ?? ''); ?></textarea>
                </div>
                
                <div class="col-12">
                    <label class="form-label">Catatan</label>
                    <textarea class="form-control" name="notes" rows="2"><?php echo htmlspecialchars($participant['notes'] ?? ''); ?></textarea>
                </div>
            </div>
            
            <div class="mt-4 d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-check-lg me-2"></i>Simpan
                </button>
                <a href="participants.php<?php echo $filterEventId ? '?event_id=' . $filterEventId : ''; ?>" class="btn btn-secondary">
                    <i class="bi bi-x-lg me-2"></i>Batal
                </a>
            </div>
        </form>
    </div>
</div>

<script>
// Auto-fill paid amount when event selected
document.querySelector('select[name="event_id"]').addEventListener('change', function() {
    const selected = this.options[this.selectedIndex];
    const price = selected.getAttribute('data-price');
    if (price && !document.getElementById('paid_amount').value.replace(/[^0-9]/g, '')) {
        document.getElementById('paid_amount').value = 'Rp ' + parseInt(price).toLocaleString('id-ID');
    }
});
</script>

<?php else: ?>
<!-- Participants List -->
<div class="row g-4 mb-4">
    <div class="col-md-3">
        <div class="stat-card primary">
            <i class="bi bi-people stat-icon"></i>
            <div class="stat-value"><?php echo number_format($summary['total']); ?></div>
            <div class="stat-label">Total Peserta</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card success">
            <i class="bi bi-check-circle stat-icon"></i>
            <div class="stat-value"><?php echo number_format($summary['paid']); ?></div>
            <div class="stat-label">Sudah Bayar</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card warning">
            <i class="bi bi-clock stat-icon"></i>
            <div class="stat-value"><?php echo number_format($summary['pending']); ?></div>
            <div class="stat-label">Menunggu</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card info">
            <i class="bi bi-cash-stack stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($summary['total_revenue']); ?></div>
            <div class="stat-label">Total Revenue</div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
        <h5 class="card-title mb-0">Daftar Peserta</h5>
        <div class="d-flex gap-2 flex-wrap">
            <select class="form-select" id="filterEvent" style="width: 150px;" onchange="filterByEvent(this.value)">
                <option value="">Semua Event</option>
                <?php foreach ($events as $evt): ?>
                <option value="<?php echo $evt['id']; ?>" <?php echo $filterEventId == $evt['id'] ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($evt['name']); ?>
                </option>
                <?php endforeach; ?>
            </select>
            <select class="form-select" id="filterStatus" style="width: 130px;" onchange="filterByStatus(this.value)">
                <option value="">Semua Status</option>
                <option value="pending" <?php echo $filterStatus === 'pending' ? 'selected' : ''; ?>>Pending</option>
                <option value="paid" <?php echo $filterStatus === 'paid' ? 'selected' : ''; ?>>Paid</option>
                <option value="refunded" <?php echo $filterStatus === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
            </select>
            <input type="text" id="searchParticipant" class="form-control" placeholder="Cari peserta..." style="width: 200px;">
            <a href="participants.php?action=create<?php echo $filterEventId ? '&event_id=' . $filterEventId : ''; ?>" class="btn btn-primary">
                <i class="bi bi-plus-lg me-2"></i>Tambah
            </a>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="participantsTable">
                <thead>
                    <tr>
                        <th>Kode</th>
                        <th>Nama</th>
                        <th>Event</th>
                        <th>Sekolah</th>
                        <th>Status</th>
                        <th>Jumlah</th>
                        <th>Terdaftar</th>
                        <th class="no-print">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($participants as $p): ?>
                    <tr>
                        <td><code><?php echo $p['registration_code']; ?></code></td>
                        <td>
                            <strong><?php echo htmlspecialchars($p['name']); ?></strong>
                            <?php if ($p['attended']): ?>
                            <span class="badge bg-success ms-1"><i class="bi bi-check"></i> Hadir</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo htmlspecialchars($p['event_name']); ?></td>
                        <td><?php echo htmlspecialchars($p['school'] ?: '-'); ?></td>
                        <td>
                            <span class="badge badge-<?php echo $p['payment_status']; ?>">
                                <?php echo ucfirst($p['payment_status']); ?>
                            </span>
                        </td>
                        <td><?php echo formatRupiah($p['paid_amount']); ?></td>
                        <td><?php echo formatDate($p['registered_at'], 'd M Y'); ?></td>
                        <td class="no-print">
                            <div class="btn-group btn-group-sm">
                                <?php if (!$p['attended'] && $p['payment_status'] === 'paid'): ?>
                                <a href="participants.php?action=attend&id=<?php echo $p['id']; ?>&csrf_token=<?php echo generateCSRFToken(); ?>" 
                                   class="btn btn-outline-success" title="Catat Kehadiran">
                                    <i class="bi bi-check-square"></i>
                                </a>
                                <?php endif; ?>
                                <a href="participants.php?action=edit&id=<?php echo $p['id']; ?>" class="btn btn-outline-primary" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <a href="participants.php?action=delete&id=<?php echo $p['id']; ?>&csrf_token=<?php echo generateCSRFToken(); ?>" 
                                   class="btn btn-outline-danger" title="Hapus"
                                   onclick="return confirmDelete()">
                                    <i class="bi bi-trash"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($participants)): ?>
                    <tr>
                        <td colspan="8" class="text-center py-5">
                            <i class="bi bi-people fs-1 text-muted"></i>
                            <p class="mt-3 text-muted">Belum ada peserta</p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function filterByEvent(eventId) {
    const url = new URL(window.location.href);
    if (eventId) {
        url.searchParams.set('event_id', eventId);
    } else {
        url.searchParams.delete('event_id');
    }
    window.location.href = url.toString();
}

function filterByStatus(status) {
    const url = new URL(window.location.href);
    if (status) {
        url.searchParams.set('status', status);
    } else {
        url.searchParams.delete('status');
    }
    window.location.href = url.toString();
}

searchTable('searchParticipant', 'participantsTable');
</script>

<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
