<?php
/**
 * Login Page
 */
require_once 'config.php';

// Redirect if already logged in
if (isLoggedIn()) {
    redirect('dashboard.php');
}

$error = '';
$debug = isset($_GET['debug']) ? true : false;
$debugInfo = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid CSRF token. Silakan refresh halaman.';
    } else {
        $email = sanitizeInput($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $remember = isset($_POST['remember']);
        
        if (empty($email) || empty($password)) {
            $error = 'Email dan password wajib diisi';
        } else {
            try {
                $db = getDB();
                
                // Check if account is locked
                $stmt = $db->prepare("SELECT * FROM users WHERE email = ? AND is_active = 1");
                $stmt->execute([$email]);
                $user = $stmt->fetch();
                
                if ($debug) {
                    $debugInfo['user_found'] = $user ? 'Yes' : 'No';
                    $debugInfo['email_searched'] = $email;
                    if ($user) {
                        $debugInfo['user_id'] = $user['id'];
                        $debugInfo['stored_hash'] = $user['password'];
                        $debugInfo['hash_length'] = strlen($user['password']);
                    }
                }
                
                if ($user) {
                    // Check lockout
                    if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
                        $remaining = ceil((strtotime($user['locked_until']) - time()) / 60);
                        $error = "Akun terkunci. Silakan coba lagi dalam {$remaining} menit.";
                    } else {
                        // Verify password
                        $passwordValid = password_verify($password, $user['password']);
                        
                        if ($debug) {
                            $debugInfo['password_verify_result'] = $passwordValid ? 'TRUE' : 'FALSE';
                            $debugInfo['input_password'] = $password;
                            $debugInfo['input_password_length'] = strlen($password);
                        }
                        
                        if ($passwordValid) {
                            // Reset login attempts
                            $stmt = $db->prepare("UPDATE users SET login_attempts = 0, locked_until = NULL, last_login = NOW() WHERE id = ?");
                            $stmt->execute([$user['id']]);
                            
                            // Set session
                            $_SESSION['user_id'] = $user['id'];
                            $_SESSION['user_name'] = $user['name'];
                            $_SESSION['user_email'] = $user['email'];
                            $_SESSION['user_role'] = $user['role'];
                            $_SESSION['login_time'] = time();
                            
                            // Set remember me cookie
                            if ($remember) {
                                $token = bin2hex(random_bytes(32));
                                setcookie('remember_token', $token, time() + 30 * 24 * 60 * 60, '/', '', false, true);
                            }
                            
                            // Log activity
                            logActivity('login', 'User logged in');
                            
                            // Get user's business
                            $stmt = $db->prepare("SELECT business_id FROM business_users WHERE user_id = ? LIMIT 1");
                            $stmt->execute([$user['id']]);
                            $business = $stmt->fetch();
                            if ($business) {
                                $_SESSION['current_business_id'] = $business['business_id'];
                            }
                            
                            setFlash('success', 'Selamat datang, ' . $user['name'] . '!');
                            redirect('dashboard.php');
                        } else {
                            // Increment login attempts
                            $attempts = $user['login_attempts'] + 1;
                            $lockedUntil = null;
                            
                            if ($attempts >= MAX_LOGIN_ATTEMPTS) {
                                $lockedUntil = date('Y-m-d H:i:s', time() + LOGIN_LOCKOUT_TIME);
                                $error = 'Terlalu banyak percobaan gagal. Akun dikunci selama 15 menit.';
                            } else {
                                $remaining = MAX_LOGIN_ATTEMPTS - $attempts;
                                $error = "Password salah. Sisa percobaan: {$remaining}";
                            }
                            
                            $stmt = $db->prepare("UPDATE users SET login_attempts = ?, locked_until = ? WHERE id = ?");
                            $stmt->execute([$attempts, $lockedUntil, $user['id']]);
                        }
                    }
                } else {
                    $error = 'Email atau password salah';
                }
            } catch (PDOException $e) {
                $error = 'Database error: ' . $e->getMessage();
                if ($debug) {
                    $debugInfo['db_error'] = $e->getMessage();
                }
            }
        }
    }
}

$pageTitle = 'Login';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?php echo APP_NAME; ?></title>
    
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .login-container {
            width: 100%;
            max-width: 420px;
            padding: 2rem;
        }
        
        .login-card {
            background: #fff;
            border-radius: 1rem;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .login-header {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            color: #fff;
            padding: 2rem;
            text-align: center;
        }
        
        .login-header i {
            font-size: 3rem;
            margin-bottom: 1rem;
        }
        
        .login-header h3 {
            font-weight: 700;
            margin: 0;
        }
        
        .login-header p {
            margin: 0.5rem 0 0;
            opacity: 0.9;
        }
        
        .login-body {
            padding: 2rem;
        }
        
        .form-floating {
            margin-bottom: 1rem;
        }
        
        .form-floating .form-control {
            border-radius: 0.5rem;
            border: 2px solid #e2e8f0;
        }
        
        .form-floating .form-control:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 0.2rem rgba(37, 99, 235, 0.25);
        }
        
        .btn-login {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            border: none;
            border-radius: 0.5rem;
            padding: 0.875rem;
            font-weight: 600;
            width: 100%;
            margin-top: 1rem;
        }
        
        .btn-login:hover {
            background: linear-gradient(135deg, #1d4ed8 0%, #1e40af 100%);
        }
        
        .demo-accounts {
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid #e2e8f0;
        }
        
        .demo-accounts h6 {
            color: #64748b;
            font-size: 0.875rem;
            margin-bottom: 0.75rem;
        }
        
        .demo-btn {
            font-size: 0.75rem;
            padding: 0.375rem 0.75rem;
        }
        
        .alert {
            border-radius: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .version {
            text-align: center;
            color: rgba(255,255,255,0.7);
            margin-top: 1rem;
            font-size: 0.75rem;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <div class="login-header">
                <i class="bi bi-wallet2"></i>
                <h3><?php echo APP_NAME; ?></h3>
                <p>Sistem Manajemen Keuangan & Event</p>
            </div>
            
            <div class="login-body">
                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i><?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($debug && !empty($debugInfo)): ?>
                <div class="alert alert-warning">
                    <h6><i class="bi bi-bug me-2"></i>Debug Info:</h6>
                    <pre class="mb-0" style="font-size: 0.75rem;"><?php print_r($debugInfo); ?></pre>
                </div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <?php echo csrfField(); ?>
                    
                    <div class="form-floating">
                        <input type="email" class="form-control" id="email" name="email" placeholder="Email" required autofocus>
                        <label for="email"><i class="bi bi-envelope me-2"></i>Email</label>
                    </div>
                    
                    <div class="form-floating">
                        <input type="password" class="form-control" id="password" name="password" placeholder="Password" required>
                        <label for="password"><i class="bi bi-lock me-2"></i>Password</label>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="remember" name="remember">
                            <label class="form-check-label" for="remember">Ingat saya</label>
                        </div>
                        <a href="forgot-password.php" class="text-decoration-none small">Lupa password?</a>
                    </div>
                    
                    <button type="submit" class="btn btn-primary btn-login">
                        <i class="bi bi-box-arrow-in-right me-2"></i>Masuk
                    </button>
                </form>
                
                <div class="demo-accounts">
                    <h6 class="text-center">Akun Demo (Cepat)</h6>
                    <div class="d-flex flex-wrap gap-2 justify-content-center">
                        <button type="button" class="btn btn-outline-primary demo-btn" onclick="fillLogin('owner@demo.com', 'demo123')">
                            Owner
                        </button>
                        <button type="button" class="btn btn-outline-secondary demo-btn" onclick="fillLogin('admin@demo.com', 'demo123')">
                            Admin
                        </button>
                        <button type="button" class="btn btn-outline-success demo-btn" onclick="fillLogin('finance@demo.com', 'demo123')">
                            Finance
                        </button>
                        <button type="button" class="btn btn-outline-info demo-btn" onclick="fillLogin('event@demo.com', 'demo123')">
                            Event Mgr
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <p class="version"><?php echo APP_NAME; ?> v<?php echo APP_VERSION; ?></p>
        
        <div class="text-center mt-3">
            <a href="reset-password.php" class="text-white text-decoration-underline small">Reset Password (Emergency)</a>
            <?php if (!$debug): ?>
            <span class="text-white mx-2">|</span>
            <a href="?debug=1" class="text-white text-decoration-underline small">Debug Mode</a>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Bootstrap 5 JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        function fillLogin(email, password) {
            document.getElementById('email').value = email;
            document.getElementById('password').value = password;
        }
    </script>
</body>
</html>
