<?php
/**
 * Events Management Page
 */
require_once 'includes/header.php';
requireRole(['owner', 'admin', 'event_manager']);

$pageTitle = 'Manajemen Event';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

$action = $_GET['action'] ?? 'list';
$eventId = $_GET['id'] ?? 0;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
        redirect('events.php');
    }
    
    $data = [
        'business_id' => $businessId,
        'name' => sanitizeInput($_POST['name'] ?? ''),
        'type' => $_POST['type'] ?? 'kompetisi',
        'description' => sanitizeInput($_POST['description'] ?? ''),
        'start_date' => $_POST['start_date'] ?? '',
        'end_date' => $_POST['end_date'] ?? '',
        'registration_open' => $_POST['registration_open'] ?: null,
        'registration_close' => $_POST['registration_close'] ?: null,
        'venue' => sanitizeInput($_POST['venue'] ?? ''),
        'capacity' => intval($_POST['capacity'] ?? 100),
        'price' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['price'] ?? 0)),
        'fixed_cost' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['fixed_cost'] ?? 0)),
        'variable_cost_per_participant' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['variable_cost_per_participant'] ?? 0)),
        'marketing_cost' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['marketing_cost'] ?? 0)),
        'operational_cost' => floatval(str_replace(['Rp', '.', ',', ' '], '', $_POST['operational_cost'] ?? 0)),
        'status' => $_POST['status'] ?? 'draft',
        'marketing_reach' => intval($_POST['marketing_reach'] ?? 0),
        'created_by' => $_SESSION['user_id']
    ];
    
    if (empty($data['name'])) {
        setFlash('error', 'Nama event wajib diisi');
    } else {
        if ($eventId) {
            // Update existing event
            $stmt = $db->prepare("
                UPDATE events SET 
                    name = ?, type = ?, description = ?, start_date = ?, end_date = ?,
                    registration_open = ?, registration_close = ?, venue = ?, capacity = ?,
                    price = ?, fixed_cost = ?, variable_cost_per_participant = ?,
                    marketing_cost = ?, operational_cost = ?, status = ?, marketing_reach = ?
                WHERE id = ? AND business_id = ?
            ");
            $stmt->execute([
                $data['name'], $data['type'], $data['description'], $data['start_date'], $data['end_date'],
                $data['registration_open'], $data['registration_close'], $data['venue'], $data['capacity'],
                $data['price'], $data['fixed_cost'], $data['variable_cost_per_participant'],
                $data['marketing_cost'], $data['operational_cost'], $data['status'], $data['marketing_reach'],
                $eventId, $businessId
            ]);
            
            // Update metrics
            updateEventMetrics($eventId);
            
            logActivity('update_event', "Updated event: {$data['name']}");
            setFlash('success', 'Event berhasil diperbarui');
        } else {
            // Create new event
            $stmt = $db->prepare("
                INSERT INTO events (
                    business_id, name, type, description, start_date, end_date,
                    registration_open, registration_close, venue, capacity,
                    price, fixed_cost, variable_cost_per_participant,
                    marketing_cost, operational_cost, status, marketing_reach, created_by
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $data['business_id'], $data['name'], $data['type'], $data['description'],
                $data['start_date'], $data['end_date'], $data['registration_open'],
                $data['registration_close'], $data['venue'], $data['capacity'],
                $data['price'], $data['fixed_cost'], $data['variable_cost_per_participant'],
                $data['marketing_cost'], $data['operational_cost'], $data['status'],
                $data['marketing_reach'], $data['created_by']
            ]);
            
            $newEventId = $db->lastInsertId();
            
            // Initialize metrics
            $stmt = $db->prepare("
                INSERT INTO event_metrics (event_id, total_participants, total_revenue, total_cost, gross_margin, margin_percent, roi, profit_per_participant)
                VALUES (?, 0, 0, ?, -?, -100, -100, 0)
            ");
            $stmt->execute([$newEventId, $data['fixed_cost'], $data['fixed_cost']]);
            
            logActivity('create_event', "Created event: {$data['name']}");
            setFlash('success', 'Event berhasil dibuat');
        }
        
        redirect('events.php');
    }
}

// Handle delete
if ($action === 'delete' && $eventId) {
    if (!validateCSRFToken($_GET['csrf_token'] ?? '')) {
        setFlash('error', 'Invalid CSRF token');
    } else {
        // Check if event has participants
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM participants WHERE event_id = ?");
        $stmt->execute([$eventId]);
        $participantCount = $stmt->fetch()['count'];
        
        if ($participantCount > 0) {
            setFlash('error', 'Event tidak dapat dihapus karena memiliki peserta');
        } else {
            $stmt = $db->prepare("DELETE FROM events WHERE id = ? AND business_id = ?");
            $stmt->execute([$eventId, $businessId]);
            
            logActivity('delete_event', "Deleted event ID: {$eventId}");
            setFlash('success', 'Event berhasil dihapus');
        }
    }
    redirect('events.php');
}

// Helper function to update event metrics
function updateEventMetrics($eventId) {
    $db = getDB();
    
    $stmt = $db->prepare("
        SELECT 
            e.fixed_cost,
            e.variable_cost_per_participant,
            COUNT(p.id) as total_participants,
            SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as registration_revenue
        FROM events e
        LEFT JOIN participants p ON e.id = p.event_id
        WHERE e.id = ?
        GROUP BY e.id
    ");
    $stmt->execute([$eventId]);
    $data = $stmt->fetch();
    
    if ($data) {
        $totalRevenue = $data['registration_revenue'];
        $totalCost = $data['fixed_cost'] + ($data['variable_cost_per_participant'] * $data['total_participants']);
        $grossMargin = $totalRevenue - $totalCost;
        $marginPercent = $totalRevenue > 0 ? ($grossMargin / $totalRevenue) * 100 : 0;
        $roi = $totalCost > 0 ? ($grossMargin / $totalCost) * 100 : 0;
        $profitPerParticipant = $data['total_participants'] > 0 ? $grossMargin / $data['total_participants'] : 0;
        
        $stmt = $db->prepare("
            INSERT INTO event_metrics (event_id, total_participants, total_revenue, total_cost, gross_margin, margin_percent, roi, profit_per_participant)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                total_participants = VALUES(total_participants),
                total_revenue = VALUES(total_revenue),
                total_cost = VALUES(total_cost),
                gross_margin = VALUES(gross_margin),
                margin_percent = VALUES(margin_percent),
                roi = VALUES(roi),
                profit_per_participant = VALUES(profit_per_participant)
        ");
        $stmt->execute([
            $eventId, $data['total_participants'], $totalRevenue, $totalCost,
            $grossMargin, $marginPercent, $roi, $profitPerParticipant
        ]);
    }
}

// Get event for edit/view
$event = null;
if ($eventId && in_array($action, ['edit', 'view'])) {
    $stmt = $db->prepare("
        SELECT e.*, 
               COUNT(p.id) as participant_count,
               SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as total_paid
        FROM events e
        LEFT JOIN participants p ON e.id = p.event_id
        WHERE e.id = ? AND e.business_id = ?
        GROUP BY e.id
    ");
    $stmt->execute([$eventId, $businessId]);
    $event = $stmt->fetch();
    
    if (!$event) {
        setFlash('error', 'Event tidak ditemukan');
        redirect('events.php');
    }
}

// Get all events with metrics
$stmt = $db->prepare("
    SELECT e.*, 
           COUNT(p.id) as participant_count,
           SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as revenue,
           em.margin_percent
    FROM events e
    LEFT JOIN participants p ON e.id = p.event_id
    LEFT JOIN event_metrics em ON e.id = em.event_id
    WHERE e.business_id = ?
    GROUP BY e.id
    ORDER BY e.created_at DESC
");
$stmt->execute([$businessId]);
$events = $stmt->fetchAll();

// Event types
$eventTypes = [
    'olimpiade' => 'Olimpiade',
    'kompetisi' => 'Kompetisi',
    'workshop' => 'Workshop',
    'seminar' => 'Seminar',
    'lainnya' => 'Lainnya'
];

$statuses = [
    'draft' => 'Draft',
    'dibuka' => 'Dibuka',
    'tertutup' => 'Tertutup',
    'selesai' => 'Selesai',
    'dibatalkan' => 'Dibatalkan'
];
?>

<?php if ($action === 'create' || ($action === 'edit' && $event)): ?>
<!-- Create/Edit Form -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="bi bi-<?php echo $action === 'create' ? 'plus-circle' : 'pencil'; ?> me-2"></i>
            <?php echo $action === 'create' ? 'Buat Event Baru' : 'Edit Event'; ?>
        </h5>
    </div>
    <div class="card-body">
        <form method="POST" action="" class="needs-validation" novalidate>
            <?php echo csrfField(); ?>
            
            <div class="row g-3">
                <!-- Basic Info -->
                <div class="col-md-8">
                    <div class="card border">
                        <div class="card-header bg-light">
                            <h6 class="mb-0"><i class="bi bi-info-circle me-2"></i>Informasi Dasar</h6>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-md-12">
                                    <label class="form-label">Nama Event <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" name="name" 
                                           value="<?php echo htmlspecialchars($event['name'] ?? ''); ?>" required>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Tipe Event</label>
                                    <select class="form-select" name="type">
                                        <?php foreach ($eventTypes as $key => $label): ?>
                                        <option value="<?php echo $key; ?>" <?php echo ($event['type'] ?? '') === $key ? 'selected' : ''; ?>>
                                            <?php echo $label; ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Status</label>
                                    <select class="form-select" name="status">
                                        <?php foreach ($statuses as $key => $label): ?>
                                        <option value="<?php echo $key; ?>" <?php echo ($event['status'] ?? 'draft') === $key ? 'selected' : ''; ?>>
                                            <?php echo $label; ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="col-12">
                                    <label class="form-label">Deskripsi</label>
                                    <textarea class="form-control" name="description" rows="3"><?php echo htmlspecialchars($event['description'] ?? ''); ?></textarea>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Tanggal Mulai <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" name="start_date" 
                                           value="<?php echo $event['start_date'] ?? ''; ?>" required>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Tanggal Selesai <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" name="end_date" 
                                           value="<?php echo $event['end_date'] ?? ''; ?>" required>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Pendaftaran Dibuka</label>
                                    <input type="date" class="form-control" name="registration_open" 
                                           value="<?php echo $event['registration_open'] ?? ''; ?>">
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Pendaftaran Ditutup</label>
                                    <input type="date" class="form-control" name="registration_close" 
                                           value="<?php echo $event['registration_close'] ?? ''; ?>">
                                </div>
                                
                                <div class="col-12">
                                    <label class="form-label">Lokasi/Venue</label>
                                    <input type="text" class="form-control" name="venue" 
                                           value="<?php echo htmlspecialchars($event['venue'] ?? ''); ?>">
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Kapasitas Peserta</label>
                                    <input type="number" class="form-control" name="capacity" id="capacity"
                                           value="<?php echo $event['capacity'] ?? 100; ?>" min="1">
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label">Harga Tiket (Rp)</label>
                                    <input type="text" class="form-control" name="price" id="price"
                                           value="<?php echo isset($event['price']) ? formatRupiah($event['price']) : 'Rp 0'; ?>"
                                           onkeyup="formatCurrency(this); calculateEventMetrics()">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Financial Info -->
                <div class="col-md-4">
                    <div class="card border">
                        <div class="card-header bg-light">
                            <h6 class="mb-0"><i class="bi bi-cash-stack me-2"></i>Informasi Keuangan</h6>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label class="form-label">Biaya Tetap (Rp)</label>
                                <input type="text" class="form-control" name="fixed_cost" id="fixed_cost"
                                       value="<?php echo isset($event['fixed_cost']) ? formatRupiah($event['fixed_cost']) : 'Rp 0'; ?>"
                                       onkeyup="formatCurrency(this); calculateEventMetrics()">
                                <small class="text-muted">Hadiah, piala, desain, dll</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Biaya Variabel/Peserta (Rp)</label>
                                <input type="text" class="form-control" name="variable_cost_per_participant" id="variable_cost_per_participant"
                                       value="<?php echo isset($event['variable_cost_per_participant']) ? formatRupiah($event['variable_cost_per_participant']) : 'Rp 0'; ?>"
                                       onkeyup="formatCurrency(this); calculateEventMetrics()">
                                <small class="text-muted">Snack, sertifikat, dll</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Biaya Marketing (Rp)</label>
                                <input type="text" class="form-control" name="marketing_cost" id="marketing_cost"
                                       value="<?php echo isset($event['marketing_cost']) ? formatRupiah($event['marketing_cost']) : 'Rp 0'; ?>"
                                       onkeyup="formatCurrency(this)">
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Biaya Operasional (Rp)</label>
                                <input type="text" class="form-control" name="operational_cost" id="operational_cost"
                                       value="<?php echo isset($event['operational_cost']) ? formatRupiah($event['operational_cost']) : 'Rp 0'; ?>"
                                       onkeyup="formatCurrency(this)">
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Jangkauan Marketing</label>
                                <input type="number" class="form-control" name="marketing_reach" id="marketing_reach"
                                       value="<?php echo $event['marketing_reach'] ?? 0; ?>">
                                <small class="text-muted">Estimasi jangkauan (orang)</small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Estimation Card -->
                    <div class="card border mt-3 bg-light">
                        <div class="card-header">
                            <h6 class="mb-0"><i class="bi bi-calculator me-2"></i>Estimasi</h6>
                        </div>
                        <div class="card-body">
                            <div class="d-flex justify-content-between mb-2">
                                <span>Revenue Estimasi:</span>
                                <strong id="expected_revenue">Rp 0</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Biaya Estimasi:</span>
                                <strong id="expected_cost">Rp 0</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Margin:</span>
                                <strong id="expected_margin">Rp 0</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Margin %:</span>
                                <strong id="margin_percent">0%</strong>
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between">
                                <span>Harga Break-even:</span>
                                <strong id="break_even_price" class="text-primary">Rp 0</strong>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="mt-4 d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-check-lg me-2"></i>Simpan
                </button>
                <a href="events.php" class="btn btn-secondary">
                    <i class="bi bi-x-lg me-2"></i>Batal
                </a>
            </div>
        </form>
    </div>
</div>

<script>
// Initial calculation
document.addEventListener('DOMContentLoaded', function() {
    calculateEventMetrics();
});
</script>

<?php elseif ($action === 'view' && $event): ?>
<!-- View Event Detail -->
<div class="row g-4">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0"><?php echo htmlspecialchars($event['name']); ?></h5>
                <div>
                    <a href="events.php?action=edit&id=<?php echo $event['id']; ?>" class="btn btn-sm btn-primary">
                        <i class="bi bi-pencil me-1"></i>Edit
                    </a>
                    <a href="events.php" class="btn btn-sm btn-secondary">Kembali</a>
                </div>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Tipe Event</p>
                        <p class="fw-medium"><?php echo $eventTypes[$event['type']] ?? $event['type']; ?></p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Status</p>
                        <span class="badge badge-<?php echo $event['status']; ?>">
                            <?php echo $statuses[$event['status']] ?? $event['status']; ?>
                        </span>
                    </div>
                    <div class="col-12">
                        <p class="mb-1 text-muted">Deskripsi</p>
                        <p><?php echo nl2br(htmlspecialchars($event['description'] ?? '-')); ?></p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Tanggal</p>
                        <p class="fw-medium">
                            <?php echo formatDate($event['start_date']); ?> - <?php echo formatDate($event['end_date']); ?>
                        </p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Lokasi</p>
                        <p class="fw-medium"><?php echo htmlspecialchars($event['venue'] ?: '-'); ?></p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Pendaftaran</p>
                        <p class="fw-medium">
                            <?php 
                            if ($event['registration_open'] && $event['registration_close']) {
                                echo formatDate($event['registration_open']) . ' - ' . formatDate($event['registration_close']);
                            } else {
                                echo '-';
                            }
                            ?>
                        </p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1 text-muted">Kapasitas</p>
                        <p class="fw-medium"><?php echo number_format($event['participant_count']); ?> / <?php echo number_format($event['capacity']); ?> peserta</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Participants List -->
        <div class="card mt-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">Daftar Peserta</h5>
                <a href="participants.php?event_id=<?php echo $event['id']; ?>" class="btn btn-sm btn-primary">
                    <i class="bi bi-plus-lg me-1"></i>Tambah Peserta
                </a>
            </div>
            <div class="card-body p-0">
                <?php
                $stmt = $db->prepare("
                    SELECT * FROM participants 
                    WHERE event_id = ? 
                    ORDER BY registered_at DESC 
                    LIMIT 10
                ");
                $stmt->execute([$eventId]);
                $participants = $stmt->fetchAll();
                ?>
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Nama</th>
                                <th>Sekolah</th>
                                <th>Status</th>
                                <th>Terdaftar</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($participants as $p): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($p['name']); ?></td>
                                <td><?php echo htmlspecialchars($p['school'] ?: '-'); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $p['payment_status']; ?>">
                                        <?php echo ucfirst($p['payment_status']); ?>
                                    </span>
                                </td>
                                <td><?php echo formatDate($p['registered_at'], 'd M Y'); ?></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($participants)): ?>
                            <tr>
                                <td colspan="4" class="text-center py-4 text-muted">Belum ada peserta</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Financial Summary -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Ringkasan Keuangan</h5>
            </div>
            <div class="card-body">
                <div class="d-flex justify-content-between mb-3">
                    <span class="text-muted">Harga Tiket</span>
                    <strong><?php echo formatRupiah($event['price']); ?></strong>
                </div>
                <div class="d-flex justify-content-between mb-3">
                    <span class="text-muted">Total Pendapatan</span>
                    <strong class="text-success"><?php echo formatRupiah($event['total_paid'] ?? 0); ?></strong>
                </div>
                <div class="d-flex justify-content-between mb-3">
                    <span class="text-muted">Biaya Tetap</span>
                    <strong><?php echo formatRupiah($event['fixed_cost']); ?></strong>
                </div>
                <div class="d-flex justify-content-between mb-3">
                    <span class="text-muted">Biaya Variabel</span>
                    <strong><?php echo formatRupiah($event['variable_cost_per_participant'] * $event['participant_count']); ?></strong>
                </div>
                <hr>
                <div class="d-flex justify-content-between mb-3">
                    <span class="text-muted">Margin</span>
                    <strong class="text-<?php echo ($event['margin_percent'] ?? 0) >= 0 ? 'success' : 'danger'; ?>">
                        <?php echo formatRupiah(($event['total_paid'] ?? 0) - $event['fixed_cost'] - ($event['variable_cost_per_participant'] * $event['participant_count'])); ?>
                    </strong>
                </div>
                <div class="d-flex justify-content-between">
                    <span class="text-muted">Margin %</span>
                    <strong class="text-<?php echo ($event['margin_percent'] ?? 0) >= 30 ? 'success' : (($event['margin_percent'] ?? 0) >= 10 ? 'warning' : 'danger'); ?>">
                        <?php echo number_format($event['margin_percent'] ?? 0, 1); ?>%
                    </strong>
                </div>
            </div>
        </div>
        
        <!-- Quick Actions -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="card-title mb-0">Aksi Cepat</h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <a href="participants.php?event_id=<?php echo $event['id']; ?>" class="btn btn-outline-primary">
                        <i class="bi bi-people me-2"></i>Kelola Peserta
                    </a>
                    <a href="transactions.php?event_id=<?php echo $event['id']; ?>" class="btn btn-outline-success">
                        <i class="bi bi-cash-stack me-2"></i>Kelola Transaksi
                    </a>
                    <a href="analytics.php?event_id=<?php echo $event['id']; ?>" class="btn btn-outline-info">
                        <i class="bi bi-graph-up me-2"></i>Lihat Analitik
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<?php else: ?>
<!-- Event List -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
        <h5 class="card-title mb-0">Daftar Event</h5>
        <div class="d-flex gap-2">
            <input type="text" id="searchEvent" class="form-control" placeholder="Cari event..." style="width: 200px;">
            <a href="events.php?action=create" class="btn btn-primary">
                <i class="bi bi-plus-lg me-2"></i>Buat Event
            </a>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="eventsTable">
                <thead>
                    <tr>
                        <th>Nama Event</th>
                        <th>Tipe</th>
                        <th>Tanggal</th>
                        <th>Peserta</th>
                        <th>Revenue</th>
                        <th>Margin</th>
                        <th>Status</th>
                        <th class="no-print">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($events as $evt): ?>
                    <tr>
                        <td>
                            <a href="events.php?action=view&id=<?php echo $evt['id']; ?>" class="text-decoration-none fw-medium">
                                <?php echo htmlspecialchars($evt['name']); ?>
                            </a>
                        </td>
                        <td><?php echo $eventTypes[$evt['type']] ?? $evt['type']; ?></td>
                        <td><?php echo formatDate($evt['start_date']); ?></td>
                        <td><?php echo number_format($evt['participant_count']); ?>/<?php echo $evt['capacity']; ?></td>
                        <td><?php echo formatRupiah($evt['revenue'] ?? 0); ?></td>
                        <td>
                            <span class="badge bg-<?php echo ($evt['margin_percent'] ?? 0) >= 30 ? 'success' : (($evt['margin_percent'] ?? 0) >= 10 ? 'warning' : 'danger'); ?>">
                                <?php echo number_format($evt['margin_percent'] ?? 0, 1); ?>%
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-<?php echo $evt['status']; ?>">
                                <?php echo $statuses[$evt['status']] ?? $evt['status']; ?>
                            </span>
                        </td>
                        <td class="no-print">
                            <div class="btn-group btn-group-sm">
                                <a href="events.php?action=view&id=<?php echo $evt['id']; ?>" class="btn btn-outline-info" title="Lihat">
                                    <i class="bi bi-eye"></i>
                                </a>
                                <a href="events.php?action=edit&id=<?php echo $evt['id']; ?>" class="btn btn-outline-primary" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <a href="events.php?action=delete&id=<?php echo $evt['id']; ?>&csrf_token=<?php echo generateCSRFToken(); ?>" 
                                   class="btn btn-outline-danger" 
                                   title="Hapus"
                                   onclick="return confirmDelete()">
                                    <i class="bi bi-trash"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($events)): ?>
                    <tr>
                        <td colspan="8" class="text-center py-5">
                            <i class="bi bi-calendar-x fs-1 text-muted"></i>
                            <p class="mt-3 text-muted">Belum ada event. <a href="events.php?action=create">Buat event pertama</a></p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
// Search functionality
searchTable('searchEvent', 'eventsTable');
</script>

<?php endif; ?>

<?php require_once 'includes/footer.php'; ?>
