-- =====================================================
-- SISTEM KEUANGAN TERPADU - DATABASE SCHEMA
-- =====================================================
-- Versi: 1.0.0
-- Author: Cita Cerdas
-- 
-- Petunjuk Penggunaan:
-- 1. Buat database baru di phpMyAdmin
-- 2. Import file ini
-- 3. Default user: admin@admin.com / admin123
-- =====================================================

SET FOREIGN_KEY_CHECKS = 0;
SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+07:00";

-- -----------------------------------------------------
-- Table: users
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `users` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(100) NOT NULL,
  `email` VARCHAR(100) NOT NULL,
  `password` VARCHAR(255) NOT NULL,
  `role` ENUM('owner','admin','finance','event_manager') NOT NULL DEFAULT 'event_manager',
  `phone` VARCHAR(20) DEFAULT NULL,
  `avatar` VARCHAR(255) DEFAULT NULL,
  `is_active` TINYINT(1) NOT NULL DEFAULT 1,
  `last_login` DATETIME DEFAULT NULL,
  `login_attempts` INT(11) NOT NULL DEFAULT 0,
  `locked_until` DATETIME DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_email` (`email`),
  KEY `idx_role` (`role`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: businesses
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `businesses` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(150) NOT NULL,
  `description` TEXT DEFAULT NULL,
  `address` TEXT DEFAULT NULL,
  `phone` VARCHAR(20) DEFAULT NULL,
  `email` VARCHAR(100) DEFAULT NULL,
  `logo` VARCHAR(255) DEFAULT NULL,
  `is_active` TINYINT(1) NOT NULL DEFAULT 1,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: business_users (many-to-many)
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `business_users` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `business_id` INT(11) UNSIGNED NOT NULL,
  `user_id` INT(11) UNSIGNED NOT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_business_user` (`business_id`, `user_id`),
  KEY `fk_bu_user` (`user_id`),
  CONSTRAINT `fk_bu_business` FOREIGN KEY (`business_id`) REFERENCES `businesses` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_bu_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: events
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `events` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `business_id` INT(11) UNSIGNED NOT NULL,
  `name` VARCHAR(200) NOT NULL,
  `type` ENUM('olimpiade','kompetisi','workshop','seminar','lainnya') NOT NULL DEFAULT 'kompetisi',
  `description` TEXT DEFAULT NULL,
  `start_date` DATE NOT NULL,
  `end_date` DATE NOT NULL,
  `registration_open` DATE DEFAULT NULL,
  `registration_close` DATE DEFAULT NULL,
  `venue` VARCHAR(255) DEFAULT NULL,
  `capacity` INT(11) NOT NULL DEFAULT 100,
  `price` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `fixed_cost` DECIMAL(15,2) NOT NULL DEFAULT 0.00 COMMENT 'Biaya tetap (hadiah, piala, desain)',
  `variable_cost_per_participant` DECIMAL(15,2) NOT NULL DEFAULT 0.00 COMMENT 'Biaya variabel per peserta',
  `marketing_cost` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `operational_cost` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `status` ENUM('draft','dibuka','tertutup','selesai','dibatalkan') NOT NULL DEFAULT 'draft',
  `marketing_reach` INT(11) DEFAULT 0,
  `banner` VARCHAR(255) DEFAULT NULL,
  `created_by` INT(11) UNSIGNED DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_business` (`business_id`),
  KEY `idx_type` (`type`),
  KEY `idx_status` (`status`),
  KEY `idx_dates` (`start_date`, `end_date`),
  CONSTRAINT `fk_event_business` FOREIGN KEY (`business_id`) REFERENCES `businesses` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_event_creator` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: participants
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `participants` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `event_id` INT(11) UNSIGNED NOT NULL,
  `registration_code` VARCHAR(50) NOT NULL,
  `name` VARCHAR(100) NOT NULL,
  `email` VARCHAR(100) DEFAULT NULL,
  `phone` VARCHAR(20) NOT NULL,
  `school` VARCHAR(150) DEFAULT NULL,
  `class` VARCHAR(50) DEFAULT NULL,
  `address` TEXT DEFAULT NULL,
  `package` VARCHAR(50) DEFAULT 'Reguler',
  `payment_status` ENUM('pending','paid','refunded','cancelled') NOT NULL DEFAULT 'pending',
  `paid_amount` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `payment_method` VARCHAR(50) DEFAULT NULL,
  `payment_date` DATETIME DEFAULT NULL,
  `payment_proof` VARCHAR(255) DEFAULT NULL,
  `attended` TINYINT(1) NOT NULL DEFAULT 0,
  `attendance_time` DATETIME DEFAULT NULL,
  `notes` TEXT DEFAULT NULL,
  `registered_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_reg_code` (`registration_code`),
  KEY `idx_event` (`event_id`),
  KEY `idx_payment_status` (`payment_status`),
  KEY `idx_name` (`name`),
  KEY `idx_phone` (`phone`),
  CONSTRAINT `fk_participant_event` FOREIGN KEY (`event_id`) REFERENCES `events` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: transactions
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `transactions` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `business_id` INT(11) UNSIGNED NOT NULL,
  `event_id` INT(11) UNSIGNED DEFAULT NULL,
  `transaction_code` VARCHAR(50) NOT NULL,
  `date` DATE NOT NULL,
  `type` ENUM('income','expense') NOT NULL,
  `category` VARCHAR(100) NOT NULL,
  `amount` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `description` TEXT DEFAULT NULL,
  `attachment` VARCHAR(255) DEFAULT NULL,
  `created_by` INT(11) UNSIGNED DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_trans_code` (`transaction_code`),
  KEY `idx_business` (`business_id`),
  KEY `idx_event` (`event_id`),
  KEY `idx_type` (`type`),
  KEY `idx_category` (`category`),
  KEY `idx_date` (`date`),
  CONSTRAINT `fk_trans_business` FOREIGN KEY (`business_id`) REFERENCES `businesses` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_trans_event` FOREIGN KEY (`event_id`) REFERENCES `events` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_trans_creator` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: event_metrics (cache untuk performa)
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `event_metrics` (
  `event_id` INT(11) UNSIGNED NOT NULL,
  `total_participants` INT(11) NOT NULL DEFAULT 0,
  `total_revenue` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `total_cost` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `gross_margin` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `margin_percent` DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  `roi` DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  `profit_per_participant` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`event_id`),
  CONSTRAINT `fk_metric_event` FOREIGN KEY (`event_id`) REFERENCES `events` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: daily_registrations
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `daily_registrations` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `event_id` INT(11) UNSIGNED NOT NULL,
  `date` DATE NOT NULL,
  `count` INT(11) NOT NULL DEFAULT 0,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_event_date` (`event_id`, `date`),
  CONSTRAINT `fk_dr_event` FOREIGN KEY (`event_id`) REFERENCES `events` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: activity_logs
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `activity_logs` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) UNSIGNED DEFAULT NULL,
  `action` VARCHAR(100) NOT NULL,
  `details` TEXT DEFAULT NULL,
  `ip_address` VARCHAR(45) DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_user` (`user_id`),
  KEY `idx_action` (`action`),
  KEY `idx_created` (`created_at`),
  CONSTRAINT `fk_log_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: settings
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `settings` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `key_name` VARCHAR(100) NOT NULL,
  `value` TEXT DEFAULT NULL,
  `description` VARCHAR(255) DEFAULT NULL,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_key` (`key_name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Table: backups
-- -----------------------------------------------------
CREATE TABLE IF NOT EXISTS `backups` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `filename` VARCHAR(255) NOT NULL,
  `file_path` VARCHAR(500) NOT NULL,
  `file_size` BIGINT(20) NOT NULL DEFAULT 0,
  `created_by` INT(11) UNSIGNED DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `fk_backup_user` (`created_by`),
  CONSTRAINT `fk_backup_user` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- -----------------------------------------------------
-- Insert Default Data
-- -----------------------------------------------------

-- Default admin user (password: admin123)
INSERT INTO `users` (`name`, `email`, `password`, `role`, `is_active`) VALUES
('Administrator', 'admin@admin.com', '$2b$12$U2OSeatZ1jgZOYKgoXDk8uc6hq3wUpqRlrdwuBTM0.iffpFZzMtoC', 'owner', 1),
('Owner Demo', 'owner@demo.com', '$2b$12$U2OSeatZ1jgZOYKgoXDk8uc6hq3wUpqRlrdwuBTM0.iffpFZzMtoC', 'owner', 1),
('Admin Demo', 'admin@demo.com', '$2b$12$U2OSeatZ1jgZOYKgoXDk8uc6hq3wUpqRlrdwuBTM0.iffpFZzMtoC', 'admin', 1),
('Finance Demo', 'finance@demo.com', '$2b$12$U2OSeatZ1jgZOYKgoXDk8uc6hq3wUpqRlrdwuBTM0.iffpFZzMtoC', 'finance', 1),
('Event Manager Demo', 'event@demo.com', '$2b$12$U2OSeatZ1jgZOYKgoXDk8uc6hq3wUpqRlrdwuBTM0.iffpFZzMtoC', 'event_manager', 1);

-- Default business
INSERT INTO `businesses` (`name`, `description`, `is_active`) VALUES
('Cita Cerdas Education', 'Lembaga pelatihan dan kompetisi akademik', 1);

-- Default settings
INSERT INTO `settings` (`key_name`, `value`, `description`) VALUES
('app_name', 'Sistem Keuangan Terpadu', 'Nama aplikasi'),
('app_logo', '', 'Logo aplikasi'),
('currency', 'IDR', 'Mata uang'),
('timezone', 'Asia/Jakarta', 'Zona waktu'),
('date_format', 'd M Y', 'Format tanggal'),
('items_per_page', '20', 'Jumlah item per halaman'),
('enable_registration', '1', 'Aktifkan pendaftaran online'),
('maintenance_mode', '0', 'Mode maintenance'),
('qris_enabled', '1', 'Aktifkan pembayaran QRIS'),
('transfer_enabled', '1', 'Aktifkan pembayaran transfer'),
('wa_notification', '1', 'Aktifkan notifikasi WhatsApp'),
('email_notification', '1', 'Aktifkan notifikasi Email');

-- -----------------------------------------------------
-- Create Views
-- -----------------------------------------------------

-- View: event_summary
CREATE OR REPLACE VIEW `v_event_summary` AS
SELECT 
  e.*,
  b.name as business_name,
  COUNT(DISTINCT p.id) as total_participants,
  COUNT(DISTINCT CASE WHEN p.payment_status = 'paid' THEN p.id END) as paid_participants,
  SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as registration_revenue,
  (e.fixed_cost + (e.variable_cost_per_participant * COUNT(DISTINCT CASE WHEN p.payment_status = 'paid' THEN p.id END))) as calculated_cost
FROM events e
LEFT JOIN businesses b ON e.business_id = b.id
LEFT JOIN participants p ON e.id = p.event_id
GROUP BY e.id;

-- View: monthly_financial_summary
CREATE OR REPLACE VIEW `v_monthly_financial` AS
SELECT 
  business_id,
  DATE_FORMAT(date, '%Y-%m') as month,
  SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as total_income,
  SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as total_expense,
  SUM(CASE WHEN type = 'income' THEN amount ELSE -amount END) as net_amount
FROM transactions
GROUP BY business_id, DATE_FORMAT(date, '%Y-%m');

SET FOREIGN_KEY_CHECKS = 1;

-- =====================================================
-- END OF DATABASE SCHEMA
-- =====================================================
