<?php
/**
 * Dashboard Page
 */
require_once 'includes/header.php';

$pageTitle = 'Dashboard';

// Get current business ID
$businessId = $_SESSION['current_business_id'] ?? 0;

$db = getDB();

// Get summary statistics
$stats = [];

// Total active events
$stmt = $db->prepare("SELECT COUNT(*) as count FROM events WHERE business_id = ? AND status IN ('draft', 'dibuka', 'tertutup')");
$stmt->execute([$businessId]);
$stats['active_events'] = $stmt->fetch()['count'];

// Total participants this month
$stmt = $db->prepare("
    SELECT COUNT(*) as count 
    FROM participants p
    JOIN events e ON p.event_id = e.id
    WHERE e.business_id = ? AND MONTH(p.registered_at) = MONTH(CURRENT_DATE()) 
    AND YEAR(p.registered_at) = YEAR(CURRENT_DATE())
");
$stmt->execute([$businessId]);
$stats['participants_this_month'] = $stmt->fetch()['count'];

// Revenue this month
$stmt = $db->prepare("
    SELECT COALESCE(SUM(amount), 0) as total 
    FROM transactions 
    WHERE business_id = ? AND type = 'income' 
    AND MONTH(date) = MONTH(CURRENT_DATE()) 
    AND YEAR(date) = YEAR(CURRENT_DATE())
");
$stmt->execute([$businessId]);
$stats['revenue_this_month'] = $stmt->fetch()['total'];

// Average margin
$stmt = $db->prepare("
    SELECT COALESCE(AVG(margin_percent), 0) as avg_margin 
    FROM event_metrics em
    JOIN events e ON em.event_id = e.id
    WHERE e.business_id = ? AND e.status = 'selesai'
");
$stmt->execute([$businessId]);
$stats['avg_margin'] = $stmt->fetch()['avg_margin'];

// Recent events
$stmt = $db->prepare("
    SELECT e.*, 
           COUNT(p.id) as participant_count,
           SUM(CASE WHEN p.payment_status = 'paid' THEN p.paid_amount ELSE 0 END) as revenue
    FROM events e
    LEFT JOIN participants p ON e.id = p.event_id
    WHERE e.business_id = ?
    GROUP BY e.id
    ORDER BY e.created_at DESC
    LIMIT 5
");
$stmt->execute([$businessId]);
$recentEvents = $stmt->fetchAll();

// Recent participants
$stmt = $db->prepare("
    SELECT p.*, e.name as event_name
    FROM participants p
    JOIN events e ON p.event_id = e.id
    WHERE e.business_id = ?
    ORDER BY p.registered_at DESC
    LIMIT 5
");
$stmt->execute([$businessId]);
$recentParticipants = $stmt->fetchAll();

// Pending payments
$stmt = $db->prepare("
    SELECT COUNT(*) as count, COALESCE(SUM(e.price), 0) as amount
    FROM participants p
    JOIN events e ON p.event_id = e.id
    WHERE e.business_id = ? AND p.payment_status = 'pending'
");
$stmt->execute([$businessId]);
$pendingPayments = $stmt->fetch();

// Chart data - Revenue vs Cost per month (last 6 months)
$stmt = $db->query("
    SELECT 
        DATE_FORMAT(date, '%Y-%m') as month,
        SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as income,
        SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as expense
    FROM transactions
    WHERE business_id = $businessId AND date >= DATE_SUB(CURRENT_DATE(), INTERVAL 6 MONTH)
    GROUP BY DATE_FORMAT(date, '%Y-%m')
    ORDER BY month
");
$monthlyData = $stmt->fetchAll();

// Chart data - Participants per event type
$stmt = $db->prepare("
    SELECT e.type, COUNT(p.id) as count
    FROM events e
    LEFT JOIN participants p ON e.id = p.event_id AND p.payment_status = 'paid'
    WHERE e.business_id = ?
    GROUP BY e.type
");
$stmt->execute([$businessId]);
$participantByType = $stmt->fetchAll();

// Top performing events
$stmt = $db->prepare("
    SELECT e.name, em.total_revenue, em.margin_percent
    FROM event_metrics em
    JOIN events e ON em.event_id = e.id
    WHERE e.business_id = ? AND e.status = 'selesai'
    ORDER BY em.gross_margin DESC
    LIMIT 5
");
$stmt->execute([$businessId]);
$topEvents = $stmt->fetchAll();

logActivity('view_dashboard', 'User viewed dashboard');
?>

<div class="row g-4">
    <!-- Summary Cards -->
    <div class="col-xl-3 col-md-6">
        <div class="stat-card primary">
            <i class="bi bi-calendar-event stat-icon"></i>
            <div class="stat-value"><?php echo $stats['active_events']; ?></div>
            <div class="stat-label">Event Aktif</div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6">
        <div class="stat-card success">
            <i class="bi bi-people stat-icon"></i>
            <div class="stat-value"><?php echo number_format($stats['participants_this_month']); ?></div>
            <div class="stat-label">Peserta Bulan Ini</div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6">
        <div class="stat-card warning">
            <i class="bi bi-cash-stack stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($stats['revenue_this_month']); ?></div>
            <div class="stat-label">Pendapatan Bulan Ini</div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6">
        <div class="stat-card info">
            <i class="bi bi-graph-up stat-icon"></i>
            <div class="stat-value"><?php echo number_format($stats['avg_margin'], 1); ?>%</div>
            <div class="stat-label">Rata-rata Margin</div>
        </div>
    </div>
</div>

<div class="row g-4 mt-2">
    <!-- Charts Row -->
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="bi bi-graph-up me-2"></i>Pendapatan vs Biaya (6 Bulan Terakhir)
                </h5>
            </div>
            <div class="card-body">
                <canvas id="revenueChart" height="300"></canvas>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-pie-chart me-2"></i>Peserta per Tipe Event
                </h5>
            </div>
            <div class="card-body">
                <canvas id="participantTypeChart" height="300"></canvas>
            </div>
        </div>
    </div>
</div>

<div class="row g-4 mt-2">
    <!-- Pending Payments Alert -->
    <?php if ($pendingPayments['count'] > 0): ?>
    <div class="col-12">
        <div class="alert alert-warning d-flex align-items-center" role="alert">
            <i class="bi bi-exclamation-triangle-fill fs-4 me-3"></i>
            <div>
                <strong><?php echo $pendingPayments['count']; ?></strong> pembayaran menunggu konfirmasi 
                (Total: <strong><?php echo formatRupiah($pendingPayments['amount']); ?></strong>)
                <a href="participants.php?status=pending" class="alert-link ms-2">Lihat Detail →</a>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Recent Events -->
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="bi bi-calendar-event me-2"></i>Event Terbaru
                </h5>
                <a href="events.php" class="btn btn-sm btn-primary">Lihat Semua</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Nama Event</th>
                                <th>Tanggal</th>
                                <th>Peserta</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recentEvents as $event): ?>
                            <tr>
                                <td>
                                    <a href="events.php?action=view&id=<?php echo $event['id']; ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($event['name']); ?>
                                    </a>
                                </td>
                                <td><?php echo formatDate($event['start_date']); ?></td>
                                <td><?php echo number_format($event['participant_count']); ?>/<?php echo $event['capacity']; ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $event['status']; ?>">
                                        <?php echo ucfirst($event['status']); ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($recentEvents)): ?>
                            <tr>
                                <td colspan="4" class="text-center py-4 text-muted">Belum ada event</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Recent Participants -->
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="bi bi-people me-2"></i>Peserta Terbaru
                </h5>
                <a href="participants.php" class="btn btn-sm btn-primary">Lihat Semua</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Nama</th>
                                <th>Event</th>
                                <th>Status</th>
                                <th>Tanggal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recentParticipants as $participant): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($participant['name']); ?></td>
                                <td><?php echo htmlspecialchars($participant['event_name']); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $participant['payment_status']; ?>">
                                        <?php echo ucfirst($participant['payment_status']); ?>
                                    </span>
                                </td>
                                <td><?php echo formatDate($participant['registered_at'], 'd M'); ?></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($recentParticipants)): ?>
                            <tr>
                                <td colspan="4" class="text-center py-4 text-muted">Belum ada peserta</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row g-4 mt-2">
    <!-- Top Performing Events -->
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="bi bi-trophy me-2"></i>Event Terbaik (Berdasarkan Margin)
                </h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Event</th>
                                <th>Total Revenue</th>
                                <th>Margin %</th>
                                <th>Performa</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($topEvents as $event): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($event['name']); ?></td>
                                <td><?php echo formatRupiah($event['total_revenue']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $event['margin_percent'] >= 30 ? 'success' : ($event['margin_percent'] >= 10 ? 'warning' : 'danger'); ?>">
                                        <?php echo number_format($event['margin_percent'], 1); ?>%
                                    </span>
                                </td>
                                <td>
                                    <div class="progress" style="height: 8px;">
                                        <div class="progress-bar bg-<?php echo $event['margin_percent'] >= 30 ? 'success' : ($event['margin_percent'] >= 10 ? 'warning' : 'danger'); ?>" 
                                             style="width: <?php echo min($event['margin_percent'], 100); ?>%"></div>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($topEvents)): ?>
                            <tr>
                                <td colspan="4" class="text-center py-4 text-muted">Belum ada event yang selesai</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Revenue vs Cost Chart
const revenueCtx = document.getElementById('revenueChart').getContext('2d');
const revenueChart = new Chart(revenueCtx, {
    type: 'bar',
    data: {
        labels: <?php echo json_encode(array_map(fn($d) => date('M Y', strtotime($d['month'] . '-01')), $monthlyData)); ?>,
        datasets: [{
            label: 'Pendapatan',
            data: <?php echo json_encode(array_map(fn($d) => $d['income'], $monthlyData)); ?>,
            backgroundColor: 'rgba(37, 99, 235, 0.8)',
            borderColor: 'rgba(37, 99, 235, 1)',
            borderWidth: 1
        }, {
            label: 'Biaya',
            data: <?php echo json_encode(array_map(fn($d) => $d['expense'], $monthlyData)); ?>,
            backgroundColor: 'rgba(239, 68, 68, 0.8)',
            borderColor: 'rgba(239, 68, 68, 1)',
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return 'Rp ' + value.toLocaleString('id-ID');
                    }
                }
            }
        }
    }
});

// Participant Type Chart
const typeCtx = document.getElementById('participantTypeChart').getContext('2d');
const typeChart = new Chart(typeCtx, {
    type: 'doughnut',
    data: {
        labels: <?php echo json_encode(array_map('ucfirst', array_column($participantByType, 'type'))); ?>,
        datasets: [{
            data: <?php echo json_encode(array_column($participantByType, 'count')); ?>,
            backgroundColor: [
                'rgba(37, 99, 235, 0.8)',
                'rgba(16, 185, 129, 0.8)',
                'rgba(245, 158, 11, 0.8)',
                'rgba(239, 68, 68, 0.8)',
                'rgba(139, 92, 246, 0.8)'
            ],
            borderWidth: 0
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom'
            }
        }
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>
