<?php
/**
 * Analytics Page
 */
require_once 'includes/header.php';

$pageTitle = 'Analitik';

$db = getDB();
$businessId = $_SESSION['current_business_id'] ?? 0;

// Get event metrics
$stmt = $db->prepare("
    SELECT e.*, em.*
    FROM events e
    LEFT JOIN event_metrics em ON e.id = em.event_id
    WHERE e.business_id = ? AND e.status = 'selesai'
    ORDER BY em.gross_margin DESC
");
$stmt->execute([$businessId]);
$eventMetrics = $stmt->fetchAll();

// Get monthly participant registration
$stmt = $db->prepare("
    SELECT 
        DATE_FORMAT(p.registered_at, '%Y-%m') as month,
        COUNT(*) as count
    FROM participants p
    JOIN events e ON p.event_id = e.id
    WHERE e.business_id = ? AND p.registered_at >= DATE_SUB(CURRENT_DATE(), INTERVAL 12 MONTH)
    GROUP BY DATE_FORMAT(p.registered_at, '%Y-%m')
    ORDER BY month
");
$stmt->execute([$businessId]);
$monthlyParticipants = $stmt->fetchAll();

// Get revenue by source
$stmt = $db->prepare("
    SELECT 
        category,
        SUM(amount) as total
    FROM transactions
    WHERE business_id = ? AND type = 'income'
    GROUP BY category
    ORDER BY total DESC
");
$stmt->execute([$businessId]);
$revenueBySource = $stmt->fetchAll();

// Get participant by event type
$stmt = $db->prepare("
    SELECT 
        e.type,
        COUNT(p.id) as count
    FROM events e
    LEFT JOIN participants p ON e.id = p.event_id AND p.payment_status = 'paid'
    WHERE e.business_id = ?
    GROUP BY e.type
");
$stmt->execute([$businessId]);
$participantsByType = $stmt->fetchAll();

// Get top performing events
$stmt = $db->prepare("
    SELECT e.name, em.total_revenue, em.gross_margin, em.margin_percent, em.roi
    FROM events e
    JOIN event_metrics em ON e.id = em.event_id
    WHERE e.business_id = ? AND e.status = 'selesai'
    ORDER BY em.gross_margin DESC
    LIMIT 10
");
$stmt->execute([$businessId]);
$topEvents = $stmt->fetchAll();

// Calculate averages
$avgMetrics = [
    'avg_revenue' => count($eventMetrics) > 0 ? array_sum(array_column($eventMetrics, 'total_revenue')) / count($eventMetrics) : 0,
    'avg_margin' => count($eventMetrics) > 0 ? array_sum(array_column($eventMetrics, 'margin_percent')) / count($eventMetrics) : 0,
    'avg_roi' => count($eventMetrics) > 0 ? array_sum(array_column($eventMetrics, 'roi')) / count($eventMetrics) : 0,
    'avg_participants' => count($eventMetrics) > 0 ? array_sum(array_column($eventMetrics, 'total_participants')) / count($eventMetrics) : 0,
];

logActivity('view_analytics', 'User viewed analytics');
?>

<div class="row g-4 mb-4">
    <div class="col-12">
        <h5 class="mb-3">Ringkasan Performa Event</h5>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="stat-card primary">
            <i class="bi bi-cash-stack stat-icon"></i>
            <div class="stat-value"><?php echo formatRupiah($avgMetrics['avg_revenue']); ?></div>
            <div class="stat-label">Rata-rata Revenue/Event</div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="stat-card success">
            <i class="bi bi-graph-up stat-icon"></i>
            <div class="stat-value"><?php echo number_format($avgMetrics['avg_margin'], 1); ?>%</div>
            <div class="stat-label">Rata-rata Margin</div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="stat-card info">
            <i class="bi bi-trophy stat-icon"></i>
            <div class="stat-value"><?php echo number_format($avgMetrics['avg_roi'], 1); ?>%</div>
            <div class="stat-label">Rata-rata ROI</div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="stat-card secondary">
            <i class="bi bi-people stat-icon"></i>
            <div class="stat-value"><?php echo number_format($avgMetrics['avg_participants'], 0); ?></div>
            <div class="stat-label">Rata-rata Peserta</div>
        </div>
    </div>
</div>

<div class="row g-4 mb-4">
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Pendaftaran Peserta per Bulan</h5>
            </div>
            <div class="card-body">
                <canvas id="participantChart" height="250"></canvas>
            </div>
        </div>
    </div>
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Peserta per Tipe Event</h5>
            </div>
            <div class="card-body">
                <canvas id="typeChart" height="250"></canvas>
            </div>
        </div>
    </div>
</div>

<div class="row g-4 mb-4">
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Sumber Pendapatan</h5>
            </div>
            <div class="card-body">
                <canvas id="revenueSourceChart" height="250"></canvas>
            </div>
        </div>
    </div>
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Margin vs ROI Event</h5>
            </div>
            <div class="card-body">
                <canvas id="scatterChart" height="250"></canvas>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">Top 10 Event Terbaik (Berdasarkan Margin)</h5>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead>
                    <tr>
                        <th>Event</th>
                        <th>Revenue</th>
                        <th>Gross Margin</th>
                        <th>Margin %</th>
                        <th>ROI</th>
                        <th>Performa</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($topEvents as $evt): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($evt['name']); ?></td>
                        <td><?php echo formatRupiah($evt['total_revenue']); ?></td>
                        <td><?php echo formatRupiah($evt['gross_margin']); ?></td>
                        <td>
                            <span class="badge bg-<?php echo $evt['margin_percent'] >= 30 ? 'success' : ($evt['margin_percent'] >= 10 ? 'warning' : 'danger'); ?>">
                                <?php echo number_format($evt['margin_percent'], 1); ?>%
                            </span>
                        </td>
                        <td><?php echo number_format($evt['roi'], 1); ?>%</td>
                        <td>
                            <div class="progress" style="height: 8px;">
                                <div class="progress-bar bg-success" style="width: <?php echo min(max($evt['margin_percent'], 0), 100); ?>%"></div>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($topEvents)): ?>
                    <tr>
                        <td colspan="6" class="text-center py-4 text-muted">Belum ada data event yang selesai</td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
// Participant Registration Chart
new Chart(document.getElementById('participantChart'), {
    type: 'line',
    data: {
        labels: <?php echo json_encode(array_map(fn($d) => date('M Y', strtotime($d['month'] . '-01')), $monthlyParticipants)); ?>,
        datasets: [{
            label: 'Pendaftaran',
            data: <?php echo json_encode(array_column($monthlyParticipants, 'count')); ?>,
            borderColor: 'rgba(37, 99, 235, 1)',
            backgroundColor: 'rgba(37, 99, 235, 0.1)',
            fill: true,
            tension: 0.4
        }]
    },
    options: {
        responsive: true,
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } }
    }
});

// Participant by Type Chart
new Chart(document.getElementById('typeChart'), {
    type: 'doughnut',
    data: {
        labels: <?php echo json_encode(array_map('ucfirst', array_column($participantsByType, 'type'))); ?>,
        datasets: [{
            data: <?php echo json_encode(array_column($participantsByType, 'count')); ?>,
            backgroundColor: [
                'rgba(37, 99, 235, 0.8)',
                'rgba(16, 185, 129, 0.8)',
                'rgba(245, 158, 11, 0.8)',
                'rgba(239, 68, 68, 0.8)',
                'rgba(139, 92, 246, 0.8)'
            ]
        }]
    },
    options: {
        responsive: true,
        plugins: { legend: { position: 'bottom' } }
    }
});

// Revenue Source Chart
new Chart(document.getElementById('revenueSourceChart'), {
    type: 'bar',
    data: {
        labels: <?php echo json_encode(array_column($revenueBySource, 'category')); ?>,
        datasets: [{
            label: 'Jumlah',
            data: <?php echo json_encode(array_column($revenueBySource, 'total')); ?>,
            backgroundColor: 'rgba(16, 185, 129, 0.8)'
        }]
    },
    options: {
        responsive: true,
        plugins: { legend: { display: false } },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return 'Rp ' + value.toLocaleString('id-ID');
                    }
                }
            }
        }
    }
});

// Scatter Chart - Margin vs ROI
new Chart(document.getElementById('scatterChart'), {
    type: 'scatter',
    data: {
        datasets: [{
            label: 'Event',
            data: <?php echo json_encode(array_map(fn($e) => ['x' => floatval($e['margin_percent']), 'y' => floatval($e['roi'])], $eventMetrics)); ?>,
            backgroundColor: 'rgba(37, 99, 235, 0.6)'
        }]
    },
    options: {
        responsive: true,
        scales: {
            x: {
                title: { display: true, text: 'Margin %' }
            },
            y: {
                title: { display: true, text: 'ROI %' }
            }
        }
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>
